# smartgrid lines monitoring (.mod file) [yvars only - define x in fn of y]

## sets and parameters
# set of buses
set V;
# type of bus: 0=generator, 1=consumer, 2=repeater (also reads V from .dat)
param bustype{V} symbolic;

# set of lines
set E within {V,V};
# line weight (also reads E from .dat)
param w{E};
# set of pairs of antiparallel arcs
set A := E union {u in V, v in V : (v,u) in E};

# set of monitoring devices
set D ordered;
# cost of monitoring device
param devcost{D};
# set of multiple-covering devices (assumes they are the first two in the set)
set DM := {member(1,D), member(2,D)};
# set of single-covering devices
set D1 := D diff DM;

# stars
set N{u in V} := {v in V : (u,v) in A};
param starsize{u in V} := card(N[u]);

## decision variables

# for d in D and (u,v) in A, v is covered by a device d installed at u 
var y{D,A} binary;

## objective function
minimize total_cost:
  sum{d in D} devcost[d] *
    sum{u in V} sum{v in N[u]} y[d,u,v] /
      (if d == "devA" then
         starsize[u]
       else if d == "devB" then
         min(2,starsize[u])
       else
         1);

## constraints

# device types
subject to consumer_repeater{u in V : bustype[u] == "gen"}:
  sum{v in N[u]} y["devB", u, v] == 0;
subject to generator{u in V : bustype[u] in {"con", "rep"}}:
  sum{v in N[u]} y["devC", u, v] == 0;
subject to repeater{u in V : bustype[u] in {"gen", "con"}}: 
  sum{v in N[u]} y["devD", u, v] == 0;
subject to consumer{u in V : bustype[u] in {"gen", "rep"}}: 
  sum{v in N[u]} y["devE", u, v] == 0;

# at most one device at each node
subject to assignment{v in V} :
  sum{d in D}
    sum{v in N[u]} y[d,u,v] /
      (if d == "devA" then
         starsize[u]
       else if d == "devB" then
         min(2,starsize[u])
       else
         1) <= 1;

# devA: every node adjacent to installed node is covered
subject to coverall{u in V}: sum{v in N[u]} y["devA",u,v] "either 0 or starsize[u]"; #[this is impossible to state linearly without an additional variable]

# devB: at most two nodes adjacent to installed node are covered
subject to covertwo{u in V} :
  sum{v in N[u]} y[member(2,D), u,v] ==
    (if starsize[u] == 1 then 1 else 2) * x[member(2,D), u];

# devC, devD, devE: exactly one node adjacent to installed node is covered
subject to coverone{d in D1, u in V} : sum{v in N[u]} y[d,u,v] == x[d, u];

# line is covered
subject to line_cover{(u,v) in E}:
  sum{d in D} y[d,u,v] + sum{e in D} y[e,v,u] >= 1;

# device at a bus
param x{D,V} binary;

