#!/usr/bin/env python
## display wordnet hyponym (=more specific words) graph

# standard imports
import string
import re
import glob
import sys
if sys.version_info < (3,):
    range = xrange
from ppretty import ppretty

# other imports
from nltk.corpus import wordnet as wn
import numpy as np
from itertools import product
import math
import networkx as nx

# plots
from mpl_toolkits.mplot3d import Axes3D
from mpl_toolkits.mplot3d import proj3d
import matplotlib.pyplot as plt
from matplotlib.path import Path
import matplotlib.patches as patches


############### FUNCTIONS ################

# WordNet: traverse (from NLTK book, p.170)
def wn_traverse(graph, start, node, depth):
    graph.depth[node.name] = depth
    for child in node.hyponyms():
        graph.add_edge(node.name, child.name)
        wn_traverse(graph, start, child, depth+1)

# WordNet: hyponym graph (from NLTK book, p.170)
def wn_hyponym_graph(start):
    G = nx.Graph()
    G.depth = {}
    wn_traverse(G, start, start, 0)
    return G

# WordNet: draw the graph (from NLTK book, p.170)
def wn_graph_draw(graph, savefilename):
    plt.figure()
    pos = nx.spring_layout(graph)
    nx.draw(graph, pos, 
            node_size = [16*graph.degree(n) for n in graph],
            node_color = [graph.depth[n] for n in graph], with_labels=False)
    labels = {i: str(graph.node.keys()[i].im_self._name.split(".")[0]) for i in range(len(graph))}
    offset = 0.01 * pos.values()[0]
    labelpos = [np.array(p) + offset for p in pos.values()]
    nx.draw_networkx_labels(graph, labelpos, labels, font_size=10)    
    plt.axis('off')
    plt.savefig(savefilename)
    plt.show()

########################################

if len(sys.argv) < 2:
    exit('need a common word on cmd line')

word = filter(str.isalpha, sys.argv[1])
savefilename = "wn_" + word + ".png"
wnword = word + ".n.01"
start = wn.synset(wnword)
graph = wn_hyponym_graph(start)
wn_graph_draw(graph, savefilename)


################### OBLIVION ######################
quit()

#print graph.node.keys()[0].im_self._name

## useful for debugging python classes!
print ppretty(graph.node.keys()[0],
              indent='    ', depth=3, width=60, seq_length=10,
              show_protected=True, show_private=False, show_static=True,
              show_properties=True, show_address=True)
