# convex MINLP for Minimum Sum-of-Squares Clustering (MSSC)

## parameters and sets

# number of points
param n >= 0, integer;
# number of clusters
param k >= 0, integer;
# number of dimensions for data
param d >= 0, integer;

set N := 1..n;
set K := 1..k;
set D := 1..d;

# data points
param p{N,D} default 0.0;

# bounds
param yL{a in D} := min{i in N} p[i,a];
param yU{a in D} := max{i in N} p[i,a];
param PU := max{i in N, h in N : i < h} sum{a in D} (p[i,a] - p[h,a])^2;

## decision variables

var y{K,a in D} >= yL[a], <= yU[a];
var x{N,K} binary;
var P{N,K} >= 0, <= PU;
var chi{N,K} >= 0, <= PU;
var xi{N,K,a in D} >= min(yL[a], 0), <= max(yU[a], 0);

## objective function

minimize mssc: sum{i in N, j in K} chi[i,j];

## constraints

# linearization of chi = Px
subject to chilin1{i in N, j in K}: chi[i,j] <= P[i,j];
subject to chilin2{i in N, j in K}: chi[i,j] >= P[i,j] - (1-x[i,j])*PU;
subject to chilin3{i in N, j in K}: chi[i,j] <= x[i,j]*PU;

# convex square norm constraint P >= ||p_i - y_j||^2
subject to sqnorm{i in N, j in K}: sum{a in D} (p[i,a] - y[j,a])^2 <= P[i,j];

# defines y_j = (sum_i p_i x_ij) / (sum_i x_ij)
subject to centroid{j in K, a in D}:
  sum{i in N} p[i,a]*x[i,j] = sum{i in N} xi[i,j,a];

# linearization of chi = yx
subject to xilin1{i in N, j in K, a in D}:
  xi[i,j,a] <= y[j,a] + (1-x[i,j])*max(abs(yL[a]),abs(yU[a]));
subject to xilin2{i in N, j in K, a in D}:
  xi[i,j,a] >= y[j,a] - (1-x[i,j])*max(abs(yL[a]),abs(yU[a]));
subject to xilin3{i in N, j in K, a in D}:
  xi[i,j,a] <= x[i,j]*max(abs(yL[a]),abs(yU[a]));
subject to xilin4{i in N, j in K, a in D}:
  xi[i,j,a] >= - x[i,j]*max(abs(yL[a]),abs(yU[a]));

# each point in exactly one cluster
subject to pointincluster{i in N}: sum{j in K} x[i,j] = 1;


