using JuMP
using LinearAlgebra

function portfolio_selection(n, R, corr, exp_ret)
    model = Model()

    @variable(model, 0 <= x[1:n] <= 1)
    @variable(model, 0 <= y[1:n] <= 1, Bin)

    # Objective: minimize covariance
    @objective(model, Min, sum(corr[i,j] * x[i] * x[j] for i in 1:n, j in 1:n))

    # Constraint: minimum expected return
    @constraint(model, sum(exp_ret[i] * x[i] for i in 1:n) >= R)

    # Constraint: sum of weights equals 1
    @constraint(model, sum(x[i] for i in 1:n) == 1)

    # Constraint: if x[i] is selected, we must buy at least 10%
    @constraint(model, min_quantity[i=1:n], x[i] >= 0.1*y[i])

    # Constraint: upper bound x[i]
    @constraint(model, max_quantity[i=1:n], x[i] <= 1*y[i])

    return model
end

using Gurobi  # or another solver
using AmplNLWriter

# Load the data from the external file
include("portfolio_selection_dat.jl")

model = portfolio_selection(n, R, corr, exp_ret)

# Set Gurobi as the solver
#set_optimizer(model, Gurobi.Optimizer)
# Set Bonmin as the solver
set_optimizer(model, () -> AmplNLWriter.Optimizer("bonmin"))

optimize!(model)

# Get the optimal weights
x_opt = value.(model[:x])
println("Optimal weights: ", x_opt)


