/*
Copyright 2020 Marianna Girlando, Lutz Straßburger.

This file is part of MOIN. MOIN is free software: you can redistribute it
and or modify it under the terms of the GNU General Public License as published
by the Free Software Foundation, version 3 of the License.

MOIN is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. See the GNU General Public License for more details. You should have
received a copy of the GNU General Public License along with MOIN. If not, see
<http://www.gnu.org/licenses/>.
*/

:- use_module(library(lists)).
:- op(500, fy, ~).     %negation
:- op(1000, xfy, ^).   %conjunction
:- op(1100, xfy, 'v'). %disjunction
:- op(1110, xfy, ->).  %implication
:- op(500, fy, '!').   %box
:- op(500, fy,'?').    %diamond

/************** Initialization *************************
To check derivability of A:

Classic modal logics, using nested sequents
:- derive(k,[],A).

Intuitionistic modal logics, using single succedent nested sequents
:- derive(i,[],A).

Intuitionistic modal logics, using multi succedent nested sequents
:- derive(m,[],A).

Add d,t,b,4,5 in the list to check derivability of A in the corresponding logic.
*******************************************************/

/* classic modal formula, one sided nested sequents */
derive(k,Axioms, Formula) :-
  clearKfiles,!,
  write('Proving in classical modal logic...'),nl,
  pre_process_k(Formula,Formula1),
  pre_process_not(Formula1,Formula2),
  prove_k(Formula,Axioms,[1,[],[(1,Formula2,+)]],Tree),nl,
  write(Tree),nl,
  tex_derivation(k,Formula,Axioms,Tree,'derivationNK.tex').

/* intuitionistic formula, single succedent */
derive(i,Axioms,Formula) :-
  clearIfiles(Formula,Axioms),!,
  write('Proving in intuitionistic modal logic, single-conclusion ...'),nl,
  pre_process_i(Formula,NewFormula),
  prove_i(Axioms,[[],1,[],[],(1,NewFormula,+),[]],Tree),nl,
  write(Tree),nl,
  tex_derivation(i,Formula,Axioms,Tree,'derivationNIKs.tex').

/* intuitionistic formula, multi succedent */
derive(m,Axioms,Formula) :-
  clearMfiles,!,
  write('Proving in intuitionistic modal logic, multi-conclusion ...'),nl,
  pre_process_i(Formula,NewFormula),
  prove_m(Axioms,[[],1,[],[],[(1,NewFormula,+)]],Tree,Counter),nl,
  write('Prooftree:'), write(Tree),nl,nl,
  write('Counter model:'), write(Counter),nl,
  counter_tree_to_list(Counter,MyList),
  remove_non_reachable(MyList,MySecondList),
  write('end remove'),nl,
  transitive_closure_list(MySecondList,TrList),
  unify_equal(TrList,CounterList),
  write('Counter model as list:'), write(CounterList),nl,
  write('done'),nl,
  tex_tikz_countermodel_m(Axioms,Formula,MySecondList,'derivationNIKm.tex'),
  (Counter\=[];tex_derivation(m,Formula,Axioms,Tree,'derivationNIKm.tex')),
  (Counter=[];tex_countermodel_m(Axioms,Formula,CounterList,'counterNIKm.tex')),
  nl.


/* failure */
derive(PS,Axioms,Formula) :-
  system_name(PS,Name),
  nl, write(Formula),write(' is NOT derivable in ' ), write(Name),write('+'),write(Axioms),!,nl.

system_name(k,'NK').
system_name(i,'NIKs').
system_name(m,'NIKm').

/************** Classic modal logic **************************/

prove_k(_,_,[_,Rel,Gamma], tree(init,[Rel,Gamma],nil,nil)) :-
  member((X,A,_),Gamma),
  member((X,(~A),_),Gamma),!,
  write('ax '),
  ter_seq_k([Rel,Gamma],1),nl.

prove_k(_,_,[_,Rel,Gamma], tree(top,[Rel,Gamma],nil,nil)) :-
  member((_,true,_),Gamma),!,
  write('ax '),
  ter_seq_k([Rel,Gamma],1),nl.

prove_k(Formula,Axioms,[Max,Rel,Gamma], tree(andK,[Rel,Gamma],SubT1,SubT2)) :-
  member((X,(A ^ B),+),Gamma),
  \+member((X,A,_),Gamma),
  \+member((X,B,_),Gamma),!,
  change_sign((X,(A^B),+),Gamma,NewGamma),
  write('and '),
  ter_seq_k([Rel,Gamma],1),nl,
  prove_k(Formula,Axioms,[Max,Rel,[(X,A,+)|NewGamma]],SubT1),
  prove_k(Formula,Axioms,[Max,Rel,[(X,B,+)|NewGamma]],SubT2).

prove_k(Formula,Axioms,[Max,Rel,Gamma], tree(orK,[Rel,Gamma],SubT,nil)) :-
  member((X,(A v B),+),Gamma),
  (\+member((X,A,_),Gamma);\+member((X,B,_),Gamma)),!,
  change_sign((X,(A v B),+),Gamma,NewGamma),
  write('or '),
  ter_seq_k([Rel,Gamma],1),nl,
  prove_k(Formula,Axioms,[Max,Rel,[(X,A,+),(X,B,+)|NewGamma]],SubT).

prove_k(Formula,Axioms,[Max,Rel,Gamma], tree(diamondK,[Rel,Gamma],SubT,nil)) :-
  member((X,(?A),+),Gamma),
  member((X,Y),Rel),
  \+member((Y,A,_),Gamma),!,
  write('diamond '),
  ter_seq_k([Rel,Gamma],1),nl,
  prove_k(Formula,Axioms,[Max,Rel,[(Y,A,+)|Gamma]],SubT).

/* Extensions */
prove_k(Formula,Axioms,[Max,Rel,Gamma], tree(diamondKT,[Rel,Gamma],SubT,nil)) :-
  member(t,Axioms),
  member((X,(?A),+),Gamma),
  \+member((X,A,_),Gamma),!,
  write('t '),
  ter_seq_k([Rel,Gamma],1),nl,
  prove_k(Formula,Axioms,[Max,Rel,[(X,A,+)|Gamma]],SubT).

prove_k(Formula,Axioms,[Max,Rel,Gamma], tree(diamondKB,[Rel,Gamma],SubT,nil)) :-
  member(b,Axioms),
  member((X,(?A),+),Gamma),
  member((Y,X),Rel),
  %(\+is4or5(Axioms);find_classical_loop(X,Rel,Gamma,[])),
  \+member((Y,A,_),Gamma),!,
  write('b '),
  ter_seq_k([Rel,Gamma],1),nl,
  prove_k(Formula,Axioms,[Max,Rel,[(Y,A,+)|Gamma]],SubT).

prove_k(Formula,Axioms,[Max,Rel,Gamma], tree(diamondK4,[Rel,Gamma],SubT,nil)) :-
  member(4,Axioms),
  member((X,(?A),+),Gamma),
  member((X,Y),Rel),
  \+member((Y,(?A),_),Gamma),!,
  write('4 '),
  ter_seq_k([Rel,Gamma],1),nl,
  prove_k(Formula,Axioms,[Max,Rel,[(Y,(?A),+)|Gamma]],SubT).

prove_k(Formula,Axioms,[Max,Rel,Gamma], tree(diamondK5,[Rel,Gamma],SubT,nil)) :-
  member(5,Axioms),
  member((X,(?A),+),Gamma),
  X > 1,
  (member((Y,X),Rel);member((X,Y),Rel);sibling(Rel,X,Y)),
  \+member((Y,(?A),_),Gamma),!,
  write('5 '),
  ter_seq_k([Rel,Gamma],1),nl,
  prove_k(Formula,Axioms,[Max,Rel,[(Y,(?A),+)|Gamma]],SubT).

prove_k(Formula,Axioms,[Max,Rel,Gamma], tree(boxK,[Rel,Gamma],SubT,nil)) :-
  member((X,(!A),+),Gamma),
  \+is_in_child(X,A,Rel,Gamma),
  (\+is4or5(Axioms);find_classical_loop(X,Rel,Gamma,[])),!,
  change_sign((X,(!A),+),Gamma,NewGamma),
  Y is Max+1,
  write('box '),
  ter_seq_k([Rel,Gamma],1),nl,
  prove_k(Formula,Axioms,[Y,[(X,Y)|Rel],[(Y,A,+)|NewGamma]],SubT).

prove_k(Formula,Axioms,[Max,Rel,Gamma], tree(diamondKD,[Rel,Gamma],SubT,nil)) :-
  member(d,Axioms),
  member((X,(?A),+),Gamma),
  (\+is4or5(Axioms);find_classical_loop(X,Rel,Gamma,[])),
  \+is_in_child(X,A,Rel,Gamma),!,
  Y is Max + 1,
  write('d '),
  ter_seq_k([Rel,Gamma],1),nl,
  prove_k(Formula,Axioms,[Y,[(X,Y)|Rel],[(Y,A,+)|Gamma]],SubT).

prove_k(Formula,Axioms,[Max,Rel,Gamma],tree(fail,[Rel,Gamma],nil,nil)) :-
  nl,nl,write('failed branch '),
  ter_seq_k([Rel,Gamma],1),nl,
  write([Max,Rel,Gamma]),nl,
  (\+is4or5(Axioms),Rel1=Rel,Gamma1=Gamma;realise_loops(Rel,Gamma,Rel1,Gamma1)),
  nl, write('after: '),write([Max,Rel1,Gamma1]),nl,
  nl,
  print_countermodel(Formula,Axioms,[Max,Rel1,Gamma1]),
  write('done '),!.

is4or5(L) :- (member(4,L);member(5,L)).

realise_loops(Rel,Gamma,Rel1,Gamma1) :-
  findall((X,Y),is_loop(Rel,Gamma,X,Y),XYs),
  write('realize loops for countermodel '),
  write(XYs),nl,
  replace_loops(Rel,Gamma,XYs,Rel1,Gamma1).

is_loop(Rel,Gamma,X,Y) :-
  member((X,(!_),+),Gamma),
  find_classical_loop(X,Rel,Gamma,Ys),
  member(Y,Ys). %, write(X),write(Y).

replace_loops(R,G,[],R,G).
replace_loops(R,G,[(X,Y)|L],R2,G2) :-
write(X),write(Y),
  remove_node_from_gamma(X,G,G1),
  replace_node_in_rel(X,Y,R,R1),
  replace_loops(R1,G1,L,R2,G2).

remove_node_from_gamma(_,[],[]).
remove_node_from_gamma(X,[(X,_,_)|G],G1) :- remove_node_from_gamma(X,G,G1).
remove_node_from_gamma(X,[(Z,A,S)|G],[(Z,A,S)|G1]) :- X\=Z, remove_node_from_gamma(X,G,G1).

replace_node_in_rel(_,_,[],[]).
replace_node_in_rel(X,Y,[(X,Z)|R],[(Y,Z)|R1]) :- replace_node_in_rel(X,Y,R,R1).
replace_node_in_rel(X,Y,[(Z,X)|R],[(Z,Y)|R1]) :- replace_node_in_rel(X,Y,R,R1).
replace_node_in_rel(X,Y,[(Z,W)|R],[(Z,W)|R1]) :- X\=Z,X\=W,replace_node_in_rel(X,Y,R,R1).



/****************************************************/
find_classical_loop(X,Rel,Gamma,Zs) :-
  write('loopcheck started'),nl,
  get_formulas_in_node(X,Gamma,ListX),
  get_ancestors(Rel,[X],An),!,
  write('Loopcheck with '),nl,
  write(ListX),write(An),nl,
  find_loop_aux(X,ListX,An,Gamma,Zs),
  write('loopcheck finished'),nl.


/* puts all ancestors of Old in relation Rel in New */
get_ancestors(_,[],[]).

get_ancestors(Rel,Old,New) :-
  member(X,Old),
  member((Y,X),Rel),
  \+member(Y,Old),
  get_ancestors(Rel,[Y|Old],New).

get_ancestors(_,Old,Old).


formulas_contained_in_node(Fms,Y,Gamma) :-
  get_formulas_in_node(Y,Gamma,ListY),
  write(Fms), write(Y), write(' --- '),write(ListY),nl,
  is_subset(Fms,ListY).

find_loop_aux(_,_,[],_,[]).
find_loop_aux(X,ListX,[Y|Ys],Gamma,Zs) :-
  (X=Y;\+formulas_contained_in_node(ListX,Y,Gamma)),
  find_loop_aux(X,ListX,Ys,Gamma,Zs).
find_loop_aux(X,ListX,[Y|Ys],Gamma,[Y|Zs]) :-
   X\=Y,
  formulas_contained_in_node(ListX,Y,Gamma),
  write(ListX), write(Y), write(Gamma),nl,
  find_loop_aux(X,ListX,Ys,Gamma,Zs).

/************** Intuitionistic modal logic, single-succedent **************************/

prove_i(_,[_,_,Rel,Lambda,Out,_], tree(false,[Rel,Lambda,Out],nil,nil)) :-
  member((_,false,_),Lambda),!,
  write('ax '),
  ter_seq_i([Rel,Lambda,Out],1),nl.

prove_i(_,[_,_,Rel,Lambda,Out,_], tree(init,[Rel,Lambda,Out],nil,nil)) :-
  member((X,A,_),Lambda),
  Out = (X,A,_),!,
  write('ax '),
  ter_seq_i([Rel,Lambda,Out],1),nl.

/* invertible rules */
prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(andL,[Rel,Lambda, Out],SubT,nil)) :-
  member((X,(A ^ B),+), Lambda),
  (\+member((X,A,_),Lambda);\+member((X,B,_),Lambda)),!,
  change_sign((X,(A ^ B),+),Lambda,LambdaNew),
  write('andL '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel, Lambda,Out,OutTmp]|Hist],Max,Rel,[(X,A,+),(X,B,+)| LambdaNew],Out,OutTmp], SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(implRs,[Rel,Lambda,Out],SubT,nil)) :-
  Out = (X,(A->B),+),!,
 %%  \+member((X,A,_), Lambda) ,!,
  write('implR '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,[(X,A,+)|Lambda],(X,B,+),[Out|OutTmp]],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(boxRs,[Rel,Lambda,Out],SubT,nil)) :-
  Out = (X, (!A),+),
  (X=1,X1=X;member(5,Axioms),X>1,X1=2;\+member(5,Axioms),X1=X),
  write('boxR '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  Y is Max +1,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Y,[(X1,Y)|Rel],Lambda,(Y, A,+),[Out|OutTmp]],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(diamondL,[Rel,Lambda,Out], SubT,nil)) :-
  member((X,(?A),+),Lambda),
  (X=1,X1=X;member(5,Axioms),X>1,X1=2;\+member(5,Axioms),X1=X),
  (\+member(4,Axioms);find_classical_loop(X,Rel,Lambda,[])),
  \+is_in_child(X1,A,Rel,Lambda),!,
  write('diamondL '),
  change_sign((X,?A,+),Lambda,LambdaNew),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  Y is Max +1,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Y,[(X1,Y)|Rel],[(Y,A,+)|LambdaNew],Out,OutTmp],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(boxL,[Rel,Lambda,Out], SubT,nil)) :-
  member((X,(!A),+), Lambda),
  member((X,Y),Rel),
  \+member((Y,A,_),Lambda),!,
  write('boxL '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,[(Y,A,+)|Lambda],Out,OutTmp],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(andR_2p,[Rel,Lambda,Out], SubT1, SubT2)) :-
  Out = (X,(A^B),+), !,
  write('andR_2p '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,Lambda,(X,A,+),[Out|OutTmp]],SubT1),
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,Lambda,(X,B,+),[Out|OutTmp]],SubT2).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(orL_2p,[Rel,Lambda,Out], SubT1, SubT2)) :-
  member((X,(A v B),+) , Lambda),
  \+member((X,A,_),Lambda),
  \+member((X,B,_),Lambda),!,
  change_sign((X,(A v B),+), Lambda,LambdaNew),
  write('orL_2p '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,[(X,A,+)|LambdaNew],Out,OutTmp],SubT1),
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,[(X,B,+)|LambdaNew],Out,OutTmp],SubT2).

/* non-invertible rules */
prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(orR1s,[Rel,Lambda,Out],SubT,nil)) :-
  Out = (X,(A v _),+),
  write('orR1 '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,Lambda,(X,A,+),[Out|OutTmp]],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(orR2s,[Rel,Lambda,Out],SubT,nil)) :-
  Out = (X,(_ v B),+),
  write('orR2 '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,Lambda,(X,B,+),[Out|OutTmp]],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(diamondR,[Rel,Lambda,Out], SubT,nil)) :-
  Out = (X, (?A),+),
  member((X,Y),Rel),
  write('diamondR '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,Lambda,(Y,A,+),[Out|OutTmp]],SubT).




/* Extensions */
prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(boxT,[Rel,Lambda,Out],SubT,nil)) :-
  member(t,Axioms),
  member((X,(!A),+),Lambda),
  \+member((X,A,_),Lambda),!,
  write('boxT '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,[(X,A,+)|Lambda],Out,OutTmp],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(diamondTs,[Rel,Lambda,Out],SubT,nil)) :-
  member(t,Axioms),
  Out = (X,(?A),+),
  write('diamondT '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,Lambda,(X,A,+),[Out|OutTmp]],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(boxD,[Rel,Lambda,Out],SubT,nil)) :-
  member(d,Axioms),
  member((X,(!A),+),Lambda),
  (X=1,X1=X;member(5,Axioms),X>1,X1=2;\+member(5,Axioms),X1=X),
  (\+member(4,Axioms);find_classical_loop(X1,Rel,Lambda,[])),
  \+ is_in_child(X1,A,Rel,Lambda),!,
  Y is Max +1,
  write('boxD '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel, Lambda,Out,OutTmp]|Hist],Y,[(X1,Y)|Rel],[(Y,A,+)|Lambda],Out,OutTmp],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(diamondDs,[Rel,Lambda,Out],SubT,nil)) :-
  member(d,Axioms),
  Out = (X, (?A),+),
  (X=1,X1=X;member(5,Axioms),X>1,X1=2;\+member(5,Axioms),X1=X),
  (\+member(4,Axioms);find_classical_loop(X1,Rel,Lambda,[])),!,
  Y is Max +1,
  write('diamondD '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Y,[(X1,Y)|Rel],Lambda,(Y,A,+),[Out|OutTmp]],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(diamond5s,[Rel,Lambda,Out],SubT,nil)) :-
  member(5,Axioms),
  Out = (X,(?A),+),
  X > 1,
  (member((Y,X),Rel);member((X,Y),Rel);sibling(Rel,X,Y)),
  \+member((Y,(?A),_),OutTmp),
  write('diamond5 '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,Lambda,(Y,(?A),+),[Out|OutTmp]],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(box5,[Rel,Lambda,Out],SubT,nil)) :-
  member(5,Axioms),
  member((X,(!A),+),Lambda),
  X > 1,
  (member((Y,X),Rel);member((X,Y),Rel);sibling(Rel,X,Y)),
  \+member((Y,(!A),_),Lambda),!,
  write('box5 '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,[(Y,(!A),+)|Lambda],Out,OutTmp],SubT).


prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(boxB,[Rel,Lambda,Out],SubT,nil)) :-
  member(b,Axioms),
  member((X,(!A),+),Lambda),
  member((Y,X),Rel),
  \+member((Y,A,_),Lambda),!,
  write('boxB '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,[(Y,A,+)|Lambda],Out,OutTmp],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(diamondBs,[Rel,Lambda,Out],SubT,nil)) :-
  member(b,Axioms),
  Out = (X, (?A),+),
  member((Y,X),Rel),!,
  write('diamondB '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,Lambda,(Y,A,+),[Out|OutTmp]],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(box4,[Rel,Lambda,Out],SubT,nil)) :-
  member(4,Axioms),
  member((X,(!A),+),Lambda),
  member((X,Y),Rel),
  \+member((Y,(!A),_),Lambda),!,
  write('box4 '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,[(Y,(!A),+)|Lambda],Out,OutTmp],SubT).

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(diamond4s,[Rel,Lambda,Out],SubT,nil)) :-
  member(4,Axioms),
  Out = (X,(?A),+),
  member((X,Y),Rel),
  \+member((Y,(?A),_),OutTmp),
  write('diamond4 '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,Lambda,(Y,(?A),+),[Out|OutTmp]],SubT).


prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(bracketsD,[Rel,Lambda,Out],SubT,nil)) :-
  member(d1,Axioms),
%  (member(4,Axioms);member(5,Axioms)),
  (member((X,(!_),+),Lambda); Out= (X,(?_),+)),
  \+member((X,_),Rel),!,
  (X=1,X1=X;member(5,Axioms),X>1,X1=2;\+member(5,Axioms),X1=X),
  %(\+member(4,Axioms);find_classical_loop(X,Rel,Lambda,[])),
  Y is Max +1,
  write('Dbrackets '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Y,[(X1,Y)|Rel],Lambda,Out,OutTmp],SubT).


/* left implication */

prove_i(Axioms,[Hist,Max,Rel,Lambda,Out,OutTmp], tree(implLs,[Rel,Lambda,Out], SubT1, SubT2)) :-
  member((X,(A->B),+),Lambda),
  write('perform intuitionistic loopcheck'),nl,
  not_occurs_hist_i([Rel,Lambda,Out,OutTmp],Hist),
  write('impL_2p '),
  ter_seq_i([Rel,Lambda,Out],1),nl,
  change_sign((X,(A->B),+),Lambda,LambdaNew),
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,[(X,B,+)|LambdaNew],Out,OutTmp],SubT1),
  prove_i(Axioms,[[[Rel,Lambda,Out,OutTmp]|Hist],Max,Rel,Lambda,(X,A,+),[]],SubT2).

/* failure */

prove_i(_,[_,_,Rel,Lambda,Out,_],_) :-
  write('failed branch '),
  ter_seq_i([Rel,Lambda,Out],1),nl,false.

/********************* Intuitionistic Multi-succedent calculi************************/

prove_m(_,[_,_,Rel,Gamma,Delta], tree(init,[Rel,Gamma,Delta],nil,nil),[]) :-
  member((X,A,_),Gamma),
  member((X,A,_),Delta),!,
  write('ax '),
  ter_seq_m([Rel,Gamma,Delta],1),nl.

prove_m(_,[_,_,Rel,Gamma,Delta], tree(init,[Rel,Gamma,Delta],nil,nil),[]) :-
  member((_,false,_),Gamma),!,
  write('ax '),
  ter_seq_m([Rel,Gamma,Delta],1),nl.

/* invertible rules  */
prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(andL,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member((X,(A ^ B),+),Gamma),
  (\+member((X,A,_),Gamma);\+member((X,B,_),Gamma)),!,
  change_sign((X,(A^B),+),Gamma,NewGamma),
  write('andL '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,[(X,A,+),(X,B,+)|NewGamma],Delta],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(orRm,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member((X,(A v B),+),Delta),
  (\+member((X,A,_),Delta);\+member((X,B,_),Delta)),!,
  change_sign((X,(A v B),+),Delta,NewDelta),
  write('orR '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,Gamma,[(X,A,+),(X,B,+)|NewDelta]],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(andR_2p,[Rel,Gamma,Delta],SubT1,SubT2),Counter) :-
  member((X,(A ^ B),+),Delta),
  \+member((X,A,_),Delta),
  \+member((X,B,_),Delta),
  change_sign((X,(A^B),+),Delta,NewDelta),
  write('andR '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,Gamma,[(X,A,+)|NewDelta]],SubT1,Counter1),
  (SubT1 \= nil,
  (write('Success for left premiss of andR, continuing proof search with right premiss.'),nl,
  prove_m(Axioms,[Hist,Max,Rel,Gamma,[(X,B,+)|NewDelta]],SubT2,Counter2),
  append(Counter1,Counter2,Counter));
  write('Failure for left premiss of andR, stopping proof search.'),nl,
  Counter = Counter1,
  write(Counter1),nl
  ),!.

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(orL_2p,[Rel,Gamma,Delta],SubT1,SubT2),Counter) :-
  member((X,(A v B),+),Gamma),
  \+member((X,A,_),Gamma),
  \+member((X,B,_),Gamma),!,
  change_sign((X,(A v B),+),Gamma,NewGamma),
  write('orL '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,[(X,A,+)|NewGamma],Delta],SubT1,Counter1),
  (SubT1 \= nil,
  (write('Success for left premiss of orL, continuing proof search with right premiss.'),nl,
  prove_m(Axioms,[Hist,Max,Rel,[(X,B,+)|NewGamma],Delta],SubT2,Counter2),
  append(Counter1,Counter2,Counter));
  write('Failure for left premiss of orL, stopping proof search.'),nl,
  Counter = Counter1,
  write(Counter1),nl
  ),!.

prove_m(Axioms,[Hist,Max,Rel,Gamma,Delta], tree(diamondL,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member((X,(?A),+),Gamma),
  (X=1,X1=X;member(5,Axioms),X>1,X1=2;\+member(5,Axioms),X1=X),
  (\+member(4,Axioms);find_classical_loop(X,Rel,Gamma,[])),
  \+is_in_child(X1,A,Rel,Gamma),!,
  write('diamondL '),
  change_sign((X,?A,+),Gamma,NewGamma),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  Y is Max +1,
  prove_m(Axioms,[Hist,Y,[(X1,Y)|Rel],[(Y,A,+)|NewGamma],Delta],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma,Delta], tree(boxL,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member((X,(!A),+),Gamma),
  member((X,Y),Rel),
  \+member((Y,A,_),Gamma),!,
  write('boxL '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,[(Y,A,+)|Gamma],Delta],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma,Delta], tree(diamondR,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member((X,(?A),+),Delta),
  member((X,Y),Rel),
  \+member((Y,A,_),Delta),!,
  write('diamondR '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,Gamma,[(Y,A,+)|Delta]],SubT,Counter).


/* Extensions */


prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(boxD,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member(d,Axioms),
  member((X,(!A),+),Gamma),
  (X=1,X1=X;member(5,Axioms),X>1,X1=2;\+member(5,Axioms),X1=X),
  (\+member(4,Axioms);find_classical_loop(X,Rel,Gamma,[])),
  \+is_in_child(X1,A,Rel,Gamma),!,
  Y is Max +1,
  write('BoxD '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Y,[(X1,Y)|Rel],[(Y,A,+)|Gamma],Delta],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(diamondDm,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member(d,Axioms),
  member((X,(?A),+),Delta),
  (X=1,X1=X;member(5,Axioms),X>1,X1=2;\+member(5,Axioms),X1=X),
  \+is_in_child(X1,A,Rel,Delta),!,
  Y is Max +1,
  write('DiamondDm '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Y,[(X1,Y)|Rel],Gamma,[(Y,A,+)|Delta]],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(boxB,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member(b,Axioms),
  member((X,(!A),+),Gamma),
  member((Y,X),Rel),
  \+member((Y,A,_),Gamma),!,
  write('BoxB '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,[(Y,A,+)|Gamma],Delta],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(diamondBm,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member(b,Axioms),
  member((X,(?A),+),Delta),
  member((Y,X),Rel),
  \+member((Y,A,_),Delta),!,
  write('DiamondBm '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,Gamma,[(Y,A,+)|Delta]],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(boxT,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member(t,Axioms),
  member((X,(!A),+),Gamma),
  \+member((X,A,_),Gamma),!,
  write('BoxTm '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,[(X,A,+)|Gamma],Delta],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(diamondTm,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member(t,Axioms),
  member((X,(?A),+),Delta),
  \+member((X,A,_),Delta),!,
  write('DiamondTm '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,Gamma,[(X,A,+)|Delta]],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(box4,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member(4,Axioms),
  member((X,(!A),+),Gamma),
  member((X,Y),Rel),
  \+member((Y,(!A),_),Gamma),!,
  write('Box4m '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,[(Y,(!A),+)|Gamma],Delta],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(diamond4m,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member(4,Axioms),
  member((X,(?A),+),Delta),
  member((X,Y),Rel),
  \+member((Y,(?A),_),Delta),!,
  write('Diamond4m '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,Gamma,[(Y,(?A),+)|Delta]],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma,Delta], tree(box5,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member(5,Axioms),
  member((X,(!A),+),Gamma),
  X > 1,
  (member((Y,X),Rel);member((X,Y),Rel);sibling(Rel,X,Y)),
  \+member((Y,(!A),_),Gamma),!,
  write('box5m '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,[(Y,(!A),+)|Gamma],Delta],SubT,Counter).

prove_m(Axioms,[Hist,Max,Rel,Gamma,Delta], tree(diamond5m,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member(5,Axioms),
  member((X,(?A),+),Delta),
  X > 1,
  (member((Y,X),Rel);member((X,Y),Rel);sibling(Rel,X,Y)),
  \+member((Y,(?A),_),Delta),!,
  write('diamond5m '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[Hist,Max,Rel,Gamma,[(Y,(?A),+)|Delta]],SubT,Counter).


/* left implication */
prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(implLm,[Rel,Gamma,Delta],SubT1,SubT2),Counter) :-
   member((X,(A -> B),+),Gamma),
   \+member((X,A,_),Delta),
   \+member((X,B,_),Gamma),
   change_sign((X,(A->B),+),Gamma,NewGamma),
   write('implL '),
   ter_seq_m([Rel,Gamma,Delta],1),nl,
   prove_m(Axioms,[Hist,Max,Rel,[(X,B,+)|NewGamma],Delta],SubT1,Counter1),
   (SubT1 \= nil,
   (write('Success for left premiss of impL, continuing proof search with right premiss.'),nl,
   prove_m(Axioms,[Hist,Max,Rel,Gamma,[(X,A,+)|Delta]],SubT2,Counter2),
   append(Counter1,Counter2,Counter));
   write('Failure for left premiss of impL, stopping proof search.'),nl,
   Counter = Counter1,
   write(Counter1),nl
   ),!.

/* rules with deletion (non invertible) */

prove_m(Axioms,[Hist,Max,Rel,Gamma, Delta], NewTree, NewCounter) :-
  %write('remove duplicates from '), write([Rel1,Gamma1,Delta1]),nl,
  %remove_all_duplicates([Rel1,Gamma1,Delta1],[Rel,Gamma,Delta]),
  write('perform intuitionistic loopcheck with '), write([Rel,Gamma,Delta]),nl,
  write('History: '),write(Hist),nl,
  not_occurs_hist_m([Rel,Gamma,Delta],Hist),
  write('no loop detected...'),nl,nl,
  bagof((Tree,Counter),(
    prove_m_implication(Axioms,[Hist,Max,Rel,Gamma, Delta],Tree,Counter);
    prove_m_box(Axioms,[Hist,Max,Rel,Gamma,Delta], Tree,Counter)), TCList),
    (member((T,[]),TCList),
     NewTree=T,NewCounter=[] ;
     NewTree=nil, extractCounter(TCList,CList), flatten(CList,FCList),NewCounter=[((Rel,Gamma,Delta),FCList)] ),!.


/* failure */
prove_m(_,[_,_,Rel,Gamma,Delta],nil,[((Rel,Gamma,Delta),[])]) :- write('failed branch '),!,
  ter_seq_m([Rel,Gamma,Delta],1),nl.


prove_m_implication(Axioms,[Hist,Max,Rel,Gamma, Delta], tree(implRm,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member((X,(A -> B),+),Delta),
  (\+member((X,A,_),Gamma);\+member((X,B,_),Delta)),!,
  write('implRm '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  prove_m(Axioms,[[[Rel,Gamma, Delta]|Hist],Max,Rel,[(X,A,+)|Gamma],[(X,B,+)]],SubT,Counter).

prove_m_box(Axioms,[Hist,Max,Rel,Gamma,Delta], tree(boxRm,[Rel,Gamma,Delta],SubT,nil),Counter) :-
  member((X,(!A),+),Delta),
  (X=1,X1=X;member(5,Axioms),X>1,X1=2;\+member(5,Axioms),X1=X),
% (X=1,X1=X;member(5,Axioms),X1=1;\+member(5,Axioms),X1=X),
  \+is_in_child(X1,A,Rel,Delta),!,
  write('boxRm '),
  ter_seq_m([Rel,Gamma,Delta],1),nl,
  Y is Max+1,
  prove_m(Axioms,[[[Rel,Gamma, Delta]|Hist],Y,[(X1,Y)|Rel],Gamma,[(Y,A,+)]],SubT,Counter).


/* auxiliary predicates for countermodel construction during proof search */

extractCounter([],[]).
extractCounter([(_,C)|L],[C|L1]) :- extractCounter(L,L1).



/********************** Auxiliary predicates **************************/
/* checks if L has a child in Lambda labelling A */
is_in_child(L,A,Rel,Lambda) :- member((L,Z),Rel), member((Z,A,_),Lambda).

/* checks if L has a child Z labelling A in the output of some sequent in the history */
output_history(L,A,Rel,Hist) :- member((L,Z),Rel), \+occurs_output((Z,A),Hist).

/* checks if L has a parent Z labelling A in the output of some sequent in the history */
output_history_b(L,A,Rel,Hist) :- member((Z,L),Rel), \+occurs_output((Z,A),Hist).

occurs_output(_,[]) :- false.
occurs_output(Out,[[_,_,Out]|_]).
occurs_output(Out,[[(_,_,H)]|T]) :- Out \= H,
  occurs_output(Out,T).

/* checks if L has a child labelling A occurring in the previous consequents, multi succedent */

output_history_m(L,A,Rel,Outs) :- member((L,Z),Rel), occurs_outs_m((Z,A,_),Outs).

occurs_outs_m(_,[]) :- false.
occurs_outs_m(Out,[L|_]) :- member(Out,L).
occurs_outs_m(Out,[L|T]) :- \+member(Out,L),
 occurs_outs_m(Out,T).

/* checks if two sequents contain the same set of formulas */
equals_sequent([Rel1,Lambda1,Out1],[Rel2,Lambda2,Out2]) :-
  Out1 = Out2,
  equals_set(Rel1,Rel2),
  equals_set(Lambda1,Lambda2).

/* checks if a sequent Seq occurs in a list of sequents,
multi conclusion variant */

not_occurs_hist_m(_,[]).
not_occurs_hist_m(Seq,[H|T]) :- \+contains_sequent_m(Seq,H),
  not_occurs_hist_m(Seq,T).
/*
not_occurs_hist_m(Seq,[H|_]) :- contains_sequent_m(Seq,H),
  write('Here is the problem:'),nl,
  write('Sequent: '),
  write(Seq), nl,
  write('Sequent in history: '),nl,
  write(H),nl,
  false.
*/
not_occurs_hist_i(_,[]).
not_occurs_hist_i(Seq,[H|T]) :- \+contains_sequent_i(Seq,H),
  not_occurs_hist_i(Seq,T).

/* checks if the first sequent is contained in the second one
multi conclusion variant */

contains_sequent_m([Rel1,Gamma1,Delta1],[Rel2,Gamma2,Delta2]) :-
  is_subsequent_tree(1,[Rel1,Gamma1,Delta1],1,[Rel2,Gamma2,Delta2]).

contains_sequent_i([Rel1,Gamma1,Out1,_],[Rel2,Gamma2,Out2,_]) :-
  is_subsequent_tree(1,[Rel1,Gamma1,[Out1]],1,[Rel2,Gamma2,[Out2]]).


/* checks if the subsequent of the first sequent rooted at N1 is
contained in the subsequent rooted at N2 in the second sequent (multi
conclusion variant) */

is_subsequent_tree(N1,[Rel1,Gamma1,Delta1],N2,[Rel2,Gamma2,Delta2]) :-
  get_formulas_in_node(N1,Gamma1,G1),
  get_formulas_in_node(N2,Gamma2,G2),
  is_subset(G1,G2),
  get_formulas_in_node(N1,Delta1,D1),
  get_formulas_in_node(N2,Delta2,D2),
  is_subset(D1,D2),
  get_children_labels(N1,Rel1,R1),
  get_children_labels(N2,Rel2,R2),
  is_subsequent_forest(R1,[Rel1,Gamma1,Delta1],R2,[Rel2,Gamma2,Delta2]).


/* checks if each of the labels in R1 is contained in one of the labels in R2 */
is_subsequent_forest([],_,_,_).
is_subsequent_forest([N|R1],[Rel1,Gamma1,Delta1],R2,[Rel2,Gamma2,Delta2]):-
  is_subsequent_forest_aux(N,[Rel1,Gamma1,Delta1],R2,[Rel2,Gamma2,Delta2]),
  is_subsequent_forest(R1,[Rel1,Gamma1,Delta1],R2,[Rel2,Gamma2,Delta2]).

is_subsequent_forest_aux(N,[Rel1,Gamma1,Delta1],[M|R2],[Rel2,Gamma2,Delta2]) :-
  (is_subsequent_tree(N,[Rel1,Gamma1,Delta1],M,[Rel2,Gamma2,Delta2]);
   is_subsequent_forest_aux(N,[Rel1,Gamma1,Delta1],R2,[Rel2,Gamma2,Delta2])).


/* checks if two lists contain the same elements*/
equals_set([],[]).
equals_set(L1,L2) :-
  is_subset(L1,L2),
  is_subset(L2,L1).

/* subset relation for lists*/
is_subset([],_).
is_subset([X|T],L) :- member(X,L),is_subset(T,L).

/* subset relation for formula lists*/
is_subset_formula_list([],_).

is_subset_formula_list([(X,A,_)|T],L) :- member((X,A,_),L),
  is_subset_formula_list(T,L).


/* changes the sign of the formula A in the antecedent Lambda */
change_sign((X,A,+),Lambda, LambdaNew) :- select((X,A,+),Lambda,Lambda1),
  LambdaNew = [(X,A,-)|Lambda1].

/* checks if X and Y are sibling, i.e. if (Z,X) and (Z,Y) belong to Rel */
sibling(Rel,X,Y) :- member((Z,X),Rel), member((Z,Y),Rel).

/* checks if X and Y have the same formulas in the formula list */
equal_node(X,Y,FList) :-
  get_formulas_in_node(X,FList,XList),
  get_formulas_in_node(Y,FList,YList),
  equals_set(XList,YList).

/* gets all formulas in the formula list that are node X, i.e., have label X,
    returns only the formulas (not the label or the signs) */
get_formulas_in_node(_,[],[]).
get_formulas_in_node(X,[(X,A,_)|Fs],[A|As]) :- get_formulas_in_node(X,Fs,As).
get_formulas_in_node(X,[(Y,_,_)|Fs],As) :- X\=Y, get_formulas_in_node(X,Fs,As).

/* gets all labels of children of the given node */
get_children_labels(_,[],[]).
get_children_labels(X,[(X,Z)|Rel],[Z|Ch]) :- get_children_labels(X,Rel,Ch).
get_children_labels(X,[(Y,_)|Rel],Ch) :- X\=Y, get_children_labels(X,Rel,Ch).



/* ************************************************************ */
/* removes label repetitions; returns a new sequent in which the two
 sequent nodes with the same formulas are identified (we keep the tree
 structure, i.e., the two have to be siblings */

find_duplicate([Rel,Gamma,Delta],X,Y) :-
  member((Z,X),Rel),
  member((Z,Y),Rel),
  X < Y,
  equal_node(X,Y,Gamma),
  equal_node(X,Y,Delta).

remove_node(_,[],[]).
remove_node(Y,[(Y,_,_)|Fs],Xs) :- remove_node(Y,Fs,Xs).
remove_node(Y,[(X,A,S)|Fs],[(X,A,S)|Xs]) :- Y\=X, remove_node(Y,Fs,Xs).

identify_nodes_in_rel([],_,_,[]).
identify_nodes_in_rel([(Y,Z)|Rel1],X,Y,[(X,Z)|Rel2]) :- identify_nodes_in_rel(Rel1,X,Y,Rel2).
identify_nodes_in_rel([(_,Y)|Rel1],X,Y,Rel2) :- identify_nodes_in_rel(Rel1,X,Y,Rel2).
identify_nodes_in_rel([(W,Z)|Rel1],X,Y,[(W,Z)|Rel2]) :- Y\=W,Y\=Z,identify_nodes_in_rel(Rel1,X,Y,Rel2).

remove_duplicate([Rel,Gamma,Delta],X,Y,[Rel1,Gamma1,Delta1]) :-
  identify_nodes_in_rel(Rel,X,Y,Rel1),
  remove_node(Y,Gamma,Gamma1),
  remove_node(Y,Delta,Delta1).

remove_all_duplicates([Rel,Gamma,Delta],[Rel,Gamma,Delta]) :-
  \+find_duplicate([Rel,Gamma,Delta],_,_).

remove_all_duplicates([Rel,Gamma,Delta],[Rel2,Gamma2,Delta2]) :-
  find_duplicate([Rel,Gamma,Delta],X,Y),
  remove_duplicate([Rel,Gamma,Delta],X,Y,[Rel1,Gamma1,Delta1]),
  remove_all_duplicates([Rel1,Gamma1,Delta1],[Rel2,Gamma2,Delta2]).


/********************** Pre-process classic modal formulas **************************/
/* Pre-process formulas for one side classic sequents */
pre_process_k(A,A) :-  atom(A).
pre_process_k(~A,(~(NewA))) :-  pre_process_k(A,NewA).
pre_process_k(A^B,((NewA)^(NewB))) :- pre_process_k(A,NewA), pre_process_k(B,NewB).
pre_process_k(A v B,((NewA) v (NewB))) :- pre_process_k(A,NewA), pre_process_k(B,NewB).
pre_process_k(?A,?(NewA)) :- pre_process_k(A,NewA).
pre_process_k(!A,!(NewA)) :- pre_process_k(A,NewA).
pre_process_k(A->false,(~(NewA))) :- pre_process_k(A,NewA).
pre_process_k(A->B,((~(NewA)) v (NewB))) :- pre_process_k(A,NewA),pre_process_k(B,NewB).

pre_process_not(A,A) :- atom(A).
pre_process_not(~A,~A) :- atom(A).
pre_process_not(~(~A),(NewA)) :- pre_process_not(A,NewA).
pre_process_not(~(A ^ B), (NewA)v(NewB)) :- pre_process_not(~(A),NewA),pre_process_not(~(B),NewB).
pre_process_not(~(A v B), (NewA)^(NewB)) :- pre_process_not(~(A),NewA),pre_process_not(~(B),NewB).
pre_process_not(~(!A), (?(NewA)) ) :- pre_process_not(~(A),NewA).
pre_process_not(~(?A), (!(NewA)) ) :- pre_process_not(~(A),NewA).
pre_process_not(A^B,(NewA)^(NewB)) :- pre_process_not(A,NewA),pre_process_not(B,NewB).
pre_process_not(A v B,(NewA)v(NewB)) :- pre_process_not(A,NewA),pre_process_not(B,NewB).
pre_process_not(!A,!(NewA)) :- pre_process_not(A,NewA).
pre_process_not(?A,?(NewA)) :- pre_process_not(A,NewA).

/* Pre-process intuitionistic formulas */
pre_process_i(A,A) :-  atom(A).
pre_process_i(A^B,NewA^NewB) :- pre_process_i(A,NewA), pre_process_i(B,NewB).
pre_process_i(A v B,NewA v NewB) :- pre_process_i(A,NewA), pre_process_i(B,NewB).
pre_process_i(A->B,NewA->NewB) :- pre_process_i(A,NewA), pre_process_i(B,NewB).
pre_process_i(?A,?NewA) :- pre_process_i(A,NewA).
pre_process_i(!A,!NewA) :- pre_process_i(A,NewA).
pre_process_i(~(A),(NewA)-> false) :- pre_process_i(A,NewA).

/********************** Latex output **************************/

clearKfiles :-
  open('counterNK.tex',write,EmptyStream),
  write(EmptyStream,''),
  close(EmptyStream),!,
  open('derivationNK.tex',write,EmptyStream1),
  write(EmptyStream1,''),
  close(EmptyStream1).

clearIfiles(Formula,Axioms) :-
  open('derivationNIKs.tex',write,EmptyStream),
  write(EmptyStream,''),
  close(EmptyStream),
  open('derivationNIKs.tex',append,Stream),
  write(Stream,'$'),
  tex_fml(Stream,Formula),
  write(Stream,'$ is not derivable in $'),
  tex_system(Stream,i),
  write(Stream, '+\\{ '),
  tex_axioms(Stream,Axioms),
  write(Stream,' \\}$ \\newline \\vspace{1cm}'),
  close(Stream).

clearMfiles :-
  open('counterNIKm.tex',write,EmptyStream),
  write(EmptyStream,''),
  close(EmptyStream),!,
  open('derivationNIKm.tex',write,EmptyStream1),
  write(EmptyStream1,''),
  close(EmptyStream1).

/* Open file out.tex */
tex_derivation(L,Formula,Axioms,Tree,Filename) :-
  open(Filename,write,EmptyStream),
  write(EmptyStream,''),
  close(EmptyStream),
  open(Filename,append,Stream),
  write(Stream,'Derivation of $'),
  tex_fml(Stream,Formula),
  write(Stream, '$ in $'),
  tex_system(Stream,L),
  write(Stream, '+\\{ '),
  tex_axioms(Stream,Axioms),
  write(Stream,' \\}$ \\newline \\vspace{1cm}'),
  tex_init(L,Stream,Tree),!,
  close(Stream).

tex_init(_,_,_,nil).
tex_init(L,Stream,Tree) :-
  write(Stream,'
  \\begin{adjustbox}{max width = \\textwidth} $ \\vlderivation{'),
  tex_write(L,Stream,Tree),!,
  write(Stream,'}$ \\end{adjustbox}').

  /* Latex tree */
tex_write(_,Stream,nil) :- write(Stream,'\\vlhy{}').
tex_write(L,Stream,tree(Rule,Seq,SubT1,nil)) :-
  write(Stream,'\\vlin{'),!,
  tex_rule(Stream,Rule),
  write(Stream,'}{}'),
  write(Stream,'{'),
  tex_seq(L,Stream,Seq,1),
  write(Stream,'}'),
  write(Stream,'{'),
  tex_write(L,Stream,SubT1),
  write(Stream,'}').
tex_write(L,Stream,tree(Rule,Seq,SubT1,SubT2)) :-
  write(Stream,'\\vliin{'),
  tex_rule(Stream,Rule),
  write(Stream,'}{}'),
  write(Stream,'{'),
  tex_seq(L,Stream,Seq,1),
  write(Stream,'}'),
  write(Stream,'{'),
  tex_write(L,Stream,SubT1),
  write(Stream,'}'),
  write(Stream,'{'),
  tex_write(L,Stream,SubT2),
  write(Stream,'}').

/* Latex classic sequent */
tex_seq(k,Stream,[Rel,Gamma],N) :-
  tex_formula_list(Stream,Gamma,N,0,''),
  write([Rel,Gamma,N]),nl,
  tex_nesting(k,Stream,Rel,N,[Rel,Gamma],0).

/* Latex intuitionistic sequent, single succedent */
tex_seq(i,Stream,[Rel,Lambda,Out],N) :-
  tex_formula_list(Stream,Lambda,N,0,'^\\bullet '),
  %%write([Rel,Lambda,Out,N]),nl,
  tex_output(Stream,Out,N,Lambda),
  tex_nesting(i,Stream,Rel,N,[Rel,Lambda,Out],0).

/* Latex intuitionistic sequent, multi succedent */
tex_seq(m,Stream,[Rel,Gamma,Delta],N) :-
  tex_formula_list(Stream,Gamma,N,0,'^\\bullet '),
  tex_output_m(Stream,Delta,N,0,Gamma),
  tex_nesting(m,Stream,Rel,N,[Rel,Gamma,Delta],0).

/* puts all formulas with label N in Gamma to the tex output, adding the polarity marker */
tex_formula_list(_,[],_,_,_).
tex_formula_list(Stream,[(N,F,+)|T],N,Count,Polarity) :-
  tex_fml(Stream,F),
  write(Stream,Polarity),
  ( (T = [(N,_,+)|_]), write(Stream,', '); write(Stream,'') ),
  NewCount is Count+1,
  tex_formula_list(Stream,T,N,NewCount,Polarity).
tex_formula_list(Stream,[(N,_,-)|T],N,Count,Polarity) :-
  ( (Count \= 0, T = [(N,_,+)|_]), write(Stream,', '); write(Stream,'') ),
  tex_formula_list(Stream,T,N,Count,Polarity).
tex_formula_list(Stream,[(M,_,_)|T],N,Count,Polarity) :- M \= N,
  ( (Count \= 0, T = [(N,_,+)|_]), write(Stream,', '); write(Stream,'') ),
  tex_formula_list(Stream,T,N,Count,Polarity).

tex_input(_,[],_,_).
tex_input(Stream,[(N,F,+)|T],N,Count) :-
  tex_fml(Stream,F),
  ( (T = [(N,_,+)|_]), write(Stream,', '); write(Stream,'') ),
  NewCount is Count+1,
  tex_input(Stream,T,N,NewCount).
tex_input(Stream,[(N,_,-)|T],N,Count) :-
  ( (Count \= 0, T = [(N,_,+)|_]), write(Stream,', '); write(Stream,'') ),
  tex_input(Stream,T,N,Count).
tex_input(Stream,[(_,_,_)|T],N,Count) :-
  ( (Count \= 0, T = [(N,_,+)|_]), write(Stream,', '); write(Stream,'') ),
  tex_input(Stream,T,N,Count).

tex_output(Stream,(N,F,_),N,Lambda) :-
  ( (member((N,_,+),Lambda)),write(Stream,', '); write(Stream,'') ),
  tex_fml(Stream,F),
  write(Stream,'^\\circ ').
tex_output(_,(N,_,_),M,_) :- N \= M.

tex_output_m(_,[],_,_,_).
tex_output_m(Stream,[(N,F,+)|T],N,Count,Gamma) :-
  ( (member((N,_,+),Gamma), Count = 0),write(Stream,', '); write(Stream,'') ),
  tex_fml(Stream,F),
  write(Stream,'^\\circ '),
  ( (T = [(N,_,+)|_]), write(Stream,', '); write(Stream,'') ),
  NewCount is Count+1,
  tex_output_m(Stream,T,N,NewCount,Gamma).
tex_output_m(Stream,[(N,_,-)|T],N,Count,Gamma) :-
  ( (Count \= 0, T = [(N,_,+)|_]), write(Stream,', '); write(Stream,'') ),
  tex_output_m(Stream,T,N,Count,Gamma).
tex_output_m(Stream,[(_,_,_)|T],N,Count,Gamma) :-
  ( (Count \= 0, T = [(N,_,+)|_]), write(Stream,', '); write(Stream,'') ),
  tex_output_m(Stream,T,N,Count,Gamma).


/* Latex nesting */
tex_nesting(_,_,[],_,_,_).

tex_nesting(k,Stream,[(N,M)|T],N,Seq,C) :-
  Seq = [_,Gamma],
  (member((N,_,+),Gamma), (write(Stream,', '), C1 is C +1); write(Stream,'')),
  write(Stream,'['),
  tex_seq(k,Stream,Seq,M),
  write(Stream,']'),
  ((C1 =0,member((N,Z),T),member((Z,_,+),Gamma)),write(Stream,', '); write(Stream,'')),
  tex_nesting(k,Stream,T,N,Seq,C).

tex_nesting(i,Stream,[(N,M)|T],N,Seq,C) :-
  Seq = [_,Lambda,(X,_,_)],
  ((member((N,_,+),Lambda);(X=N)), (write(Stream,', '), C1 is C +1) ; write(Stream,'')),
  write(Stream,'['),
  tex_seq(i,Stream,Seq,M),
  write(Stream,']'),
  ((C1 =0,member((N,Z),T),(member((Z,_,+),Lambda);X= Z)),write(Stream,', '); write(Stream,'')),
  tex_nesting(i,Stream,T,N,Seq,C).

tex_nesting(m,Stream,[(N,M)|T],N,Seq,C) :-
  Seq = [_,Gamma,Delta],
  ((member((N,_,+),Gamma);member((N,_,+),Delta)), (write(Stream,', '), C1 is C +1); write(Stream,'')),
  write(Stream,'['),
  tex_seq(m,Stream,Seq,M),
  write(Stream,']'),
  ((C1 =0,member((N,Z),T),(member((Z,_,+),Gamma);member((N,_,+),Delta))),write(Stream,', '); write(Stream,'')),
  tex_nesting(m,Stream,T,N,Seq,C).

tex_nesting(L,Stream,[(_,_)|T],N,Seq,C) :-
  tex_nesting(L,Stream,T,N,Seq,C).


/***************** Countermodel, K ********************************/


/* Latex printing a countermodel */
print_countermodel(Formula,Axioms,Seq) :-
  open('counterNK.tex',write,EmptyStream),
  write(EmptyStream,''),
  close(EmptyStream),!,
  open('counterNK.tex',append,Stream),
  print_countermodel_k(Stream,Formula,Axioms,Seq),!,
  close(Stream).

/* Prints out a countermodel */
print_countermodel_k(Stream,Formula,Axioms,[Max,Rel,Gamma]):-
  write('Classical Countermodel'),nl,
  write(Stream,'\\noindent Classical $\\{'),
  tex_axioms(Stream,Axioms),
  write(Stream,'\\}$-Countermodel of : $'),
  tex_fml(Stream,Formula),
  write(Stream,'$\\newline'),
  write(Stream,'$\\mathcal{W} = \\{'),
  world_list([[Rel,Gamma,[]]],Worlds),
  sort(Worlds,SortedWorlds),
  write('Worlds: '),
  write(SortedWorlds),nl,
  tex_worlds(Stream,SortedWorlds,0),
  write(Stream,'\\}$ \\newline'),
  write(Stream,'$\\rel = \\{'),
  closure(Axioms,Rel,NewRel),
  tex_rels(Stream,NewRel,0,'R',8),
  write(Stream,'\\}$ \\newline'),
  print_valuation(Stream,1,Max,Gamma),
  write(Stream,'\\newline'),
  print_all_formulas_gamma(Stream,1,Max,Gamma),
  write('end countermodel'),nl.

/* Returns the closure of Rel according to the Axioms */
closure(_,[],[]).
closure(Axioms,Rel,NewRel) :-
  update(Axioms,Rel,NewRel),
  (Rel \= NewRel,closure(Axioms,NewRel,Rel); true).

update(Axioms,Rel,SortedFinal) :-
  ((member(d,Axioms),\+member(t,Axioms)),closure_ser(Rel,Rel1);Rel1 = Rel),
  (member(t,Axioms),closure_ref(Rel1,Rel2);Rel2 = Rel1),
  (member(b,Axioms),closure_sym(Rel2,Rel3);Rel3 = Rel2),
  (member(4,Axioms),closure_tr(Rel3,Rel4);Rel4 = Rel3),
  (member(5,Axioms),closure_euc(Rel4,Final);Final = Rel4),
  sort(Final,SortedFinal).

closure_ser([],[]).
closure_ser(Rel,NewRel) :-
  member((_,X),Rel),
  \+member((X,_),Rel),!,
  closure_ser([(X,X)|Rel],NewRel).
closure_ser(Rel,Rel):- write('Serial saturation completed'),nl.

closure_ref([],[]).
closure_ref(Rel,NewRel) :-
  (member((X,_),Rel);member((_,X),Rel)),
  \+ member((X,X),Rel),!,
  closure_ref([(X,X)|Rel],NewRel).
closure_ref(Rel,Rel):- write('Reflexive saturation completed'),nl.

closure_sym([],[]).
closure_sym(Rel,NewRel) :-
  member((X,Y),Rel),
  \+ member((Y,X),Rel),!,
  closure_sym([(Y,X)|Rel],NewRel).
closure_sym(Rel,Rel):- write('Symmetric saturation completed'),nl.

closure_tr([],[]).
closure_tr(Rel,NewRel) :-
  member((X,Y),Rel),
  member((Y,Z),Rel),
  \+ member((X,Z),Rel),!,
  closure_tr([(X,Z)|Rel],NewRel).
closure_tr(Rel,Rel):- write('Transitive saturation completed'),nl.

closure_euc([],[]).
closure_euc(Rel,NewRel) :-
  member((X,Y),Rel),
  member((X,Z),Rel),
  (\+member((Y,Z),Rel);\+member((Z,Y),Rel)),!,
  ((\+member((Y,Z),Rel),\+member((Z,Y),Rel)),closure_euc([(Y,Z),(Z,Y)|Rel],NewRel);true),
  ((\+member((Y,Z),Rel),member((Z,Y),Rel)),closure_euc([(Y,Z)|Rel],NewRel);true),
  ((member((Y,Z),Rel),\+member((Z,Y),Rel)),closure_euc([(Z,Y)|Rel],NewRel);true).
closure_euc(Rel,Rel):- write('Euclidean saturation completed'),nl.



/* prints out Rel */
tex_Rel_init(Stream,[]) :-
  write(Stream,'\\varnothing').
tex_Rel_init(Stream,Rel) :-
  write(Stream,' \\{'),
  length(Rel,Length),
  tex_list(Stream,Rel,Length),
  write(Stream,'\\}').


tex_list(_, [],_).
tex_list(Stream, [H|T],Length) :-
  Length > 1,
  tex_rel(Stream,H),write(Stream,', '),
  Length1 is Length -1,
  tex_list(Stream,T,Length1).
tex_list(Stream, [H|T],Length) :-
  Length =< 1,
  tex_rel(Stream,H),
  tex_list(Stream,T,Length).

tex_rel(Stream,(X,Y)) :- write(Stream,X), write(Stream,'R'),write(Stream,Y).


/* Prints valuation: atomic formulas in Gamma */
print_valuation(Stream,N,Max,Gamma) :-
  N =< Max,
  occurs_negated_atom(N,Gamma),
  write(Stream,'$ \\llbracket'),
  write(Stream,N),
  write(Stream,' \\rrbracket = \\{'),
  print_negatoms_k(Stream,N,Gamma),
  write(Stream,' \\}$'),
  write(Stream,'\\newline'),
  M is N +1,
  print_valuation(Stream,M,Max,Gamma).
print_valuation(Stream,N,Max,Gamma) :-
  N =< Max,
  \+occurs_negated_atom(N,Gamma),
  write(Stream, '$ \\llbracket'),
  write(Stream,N),
  write(Stream,'\\rrbracket = \\varnothing $'),
  write(Stream,'\\newline'),
  M is N +1,
  print_valuation(Stream,M,Max,Gamma).
print_valuation(_,N,Max,_) :- N > Max.

occurs_negated_atom(_,[]) :- false.
occurs_negated_atom(X,[(X,A,_)|_]) :- negatom(A).
occurs_negated_atom(X,[(_,_,_)|T]) :- occurs_negated_atom(X,T).

print_negatoms_k(_,_,[]).
print_negatoms_k(Stream,X,[(X,~A,_)|T]) :-
  atom(A),
  write(Stream,A),
  (occurs_negated_atom(X,T), write(Stream,', ');write(Stream,'') ),
  print_negatoms_k(Stream,X,T).
print_negatoms_k(Stream,X,[(_,_,_)|T]) :-
  print_negatoms_k(Stream,X,T).


/* Prints valuation: all formulas in Gamma */

print_all_formulas_gamma(_,_,_,[]).
print_all_formulas_gamma(Stream,N,Max,Gamma):-
  N =< Max,
  member((N,_,_),Gamma),
  print_gamma(Stream,N,Gamma),
  write(Stream,'\\newline'),
  M is N +1,
  print_all_formulas_gamma(Stream,M,Max,Gamma).
print_all_formulas_gamma(Stream,N,Max,Gamma):-
  N =< Max,
  \+member((N,_,_),Gamma),
  write(Stream,'\\newline'),
  M is N +1,
  print_all_formulas_gamma(Stream,M,Max,Gamma).
print_all_formulas_gamma(_,N,Max,_):-
  N > Max.

print_gamma(_,_,[]).
print_gamma(Stream,N,[(N,A,_)|T]) :-
  write(Stream,'$ '),
  write(Stream,N),
  write(Stream,' \\nVdash '),
  tex_fml(Stream,A),
  write(Stream,' $'),
  (member((N,_,_),T),write(Stream,', ') ;write(Stream,'') ),
  print_gamma(Stream,N,T).
print_gamma(Stream,N,[(X,_,_)|T]) :-
  N \= X,
  print_gamma(Stream,N,T).


/******************** Countermodel, IK, multi succedent **************************/

/* Countermodel NIKm Preprocessing : assign a number to each sequent *************/

counter_tree_to_list([],[]) :- write('empty countermodel').

counter_tree_to_list(Counter,(FutureList,LayerList)) :-
  counter_forest_to_list_aux(1,Counter,FutureList,LayerList,_,_).

/*counter_tree_to_list_aux(N,[Root],[(N,Root)],M) :- M is N+1.*/

counter_tree_to_list_aux(N,(Root,Children),FutureList,[(N,Root)|LayerList],M) :-
   N1 is N+1,
   counter_forest_to_list_aux(N1,Children,FutureList1,LayerList,M,ChildrenRoots),
   connect_to_children(N,ChildrenRoots,FutureList2),
   append(FutureList1,FutureList2,FutureList).

counter_forest_to_list_aux(N,[],[],[],N,[]).

counter_forest_to_list_aux(N,[Tree|TreeList],FutureList,LayerList,M,[N|NRoots]) :-
  counter_tree_to_list_aux(N,Tree,FutureList1,LayerList1,N1),
  counter_forest_to_list_aux(N1,TreeList,FutureList2,LayerList2,M,NRoots),
  append(LayerList1,LayerList2,LayerList),
  append(FutureList1,FutureList2,FutureList).


connect_to_children(_,[],[]).
connect_to_children(N,[C|Cs],[(N,C)|NewList]) :- connect_to_children(N,Cs,NewList).

/* Countermodel NIKm Preprocessing : remove unreachable part **********************/

remove_non_reachable((FRel,LL1),(FRel,LL2)) :-
  saturate_reachable_futures(FRel,[1],Worlds),
  remove_others(LL1,Worlds,LL2).

saturate_reachable_futures(FRel,Ws,Ws2) :-
  write('saturate_reachable_futures'),nl,
  member(W,Ws),
  member((W,V),FRel),
  \+member(V,Ws),!,
  saturate_reachable_futures(FRel,[V|Ws],Ws2).

saturate_reachable_futures(_,Ws,Ws).

remove_others([],_,[]).

remove_others([(W,Rel,Gamma,Delta)|LL],Worlds,[(W,Rel,Gamma,Delta)|LL2]) :-
  write('remove'),nl,
  member(W,Worlds),!,
  remove_others(LL,Worlds,LL2).

remove_others([(W,_,_,_)|LL],Worlds,LL2) :-
write('remove'),nl,
  \+member(W,Worlds),!,
  remove_others(LL,Worlds,LL2).


/* Countermodel NIKm Preprocessing : transitive closure for unify_equal **********************/
transitive_closure_list((FRel,Worlds),(TrFRel,Worlds)) :-
  write('Start transitive closure'),nl,
  closure_tr(FRel, TrFRel),!,
  write('Finished transitive closure'),nl.


/* Countermodel NIKm Preprocessing :
if (X,Y) is in FutureRel, and if the formulas of Y are contained in the formulas of X,
then add (Y,X) in FutureRel **********************/

unify_equal((FRel,Worlds),(NewFRel,Worlds)) :-
  %write('Start unify'), nl,
  %closure_tr(FRel, TrFRel),
  %write('Finished transitive closure'),
  find_equal(FRel,Worlds,Equals),!,
  append(FRel,Equals,NewFRel).

find_equal([],_,[]).

find_equal([(X,Y)|T],Worlds,[(Y,X)|Equals]) :-
  find_sequent(X,Worlds,SeqX),
  %write('X is: '),
  %write(SeqX),nl,
  find_sequent(Y,Worlds,SeqY),
  %write('Y is: '),
  %write(SeqY),nl,
  contains_sequent_m(SeqY,SeqX),!,
  find_equal(T,Worlds,Equals).

find_equal([_|T],Worlds,Equals) :-
  find_equal(T,Worlds,Equals).

find_sequent(X,[(X,Rel,Gamma,Delta)|_],[Rel,Gamma,Delta]).

find_sequent(X,[(Z,_,_,_)|T],Seq) :-
  X \= Z,!,
  find_sequent(X,T,Seq).

find_sequent(X,[(Z,_,_,_)|[]],[]) :-
  X \= Z.


/* Countermodel NIKm: tikz tree ***************************************************/

tex_tikz_countermodel_m(Axioms,Formula,(FutureRel,Nodes),Filename) :-
  open(Filename,write,EmptyStream),
  write(EmptyStream,''),
  close(EmptyStream),
  open(Filename,append,Stream),
  write(Stream,'$'),
  tex_fml(Stream,Formula),
  write(Stream,'$ is not derivable in $'),
  tex_system(Stream,m),
  write(Stream, '+\\{ '),
  tex_axioms(Stream,Axioms),
  write(Stream,' \\}$ \\newline \\medskip \n\n'),
  write(Stream,'This is the structure of the birelational Kripke frame, where each vertex corresponds to a classical single-relation Kripke frame that is obtained from the sequents below:\\newline\\medskip  \\par '),
  write(Stream,'\\begin{adjustbox}{max width = \\textwidth}'),
  write(Stream,'\\tikz \\graph [nodes={draw, circle},grow up=2cm, branch right=1cm] {'),
  tikz_tree(Stream,FutureRel,1),
  write(Stream,' }; \\end{adjustbox} \\newline \\medskip \\par '),
%  write(Stream,'$ \\begin{array}{r@{:\\;}l}'),
  write(Stream,'\\begin{longtable}[l]{r l}'),
  list_countermodel_sequents(Stream,Nodes),
%  write(Stream,'\\end{array} $ \\newline  \\bigskip \\newline'),
  write(Stream,'\\end{longtable} \\bigskip '),
  close(Stream).

tikz_tree(Stream,FRel,N) :-
  write(Stream,N),
  write(Stream,' -- {'),
  findall(M,(member((N,M),FRel),N\=M),Children),
  tikz_forest(Stream,FRel,Children),
  write(Stream,' -- }, ').

tikz_forest(_,_,[]).
tikz_forest(Stream,FRel,[N|Ns]) :-
   tikz_tree(Stream,FRel,N),
   tikz_forest(Stream,FRel,Ns).

list_countermodel_sequents(_,[]).
list_countermodel_sequents(Stream,[(N,Rel,Gamma,Delta)|Nodes]) :-
  write(Stream,N),
  write(Stream,': & $'),
  tex_seq(m,Stream,[Rel,Gamma,Delta],1),
  write(Stream,'$ \\\\'),
  list_countermodel_sequents(Stream,Nodes).

/* Countermodel NIKm: printing the countermodel **********************************/


tex_countermodel_m(Axioms,Formula,(FRel,List),Filename) :-
  couple_labels(List,CoupledList),!,
  world_list(CoupledList,WorldList),
  sort(WorldList,Worlds),
  write('List of worlds :'),
  write(Worlds),nl,
  future_rels(Worlds,FRel,NewF),
  sort(NewF,FutureRels),
  write('List of future relations'),
  write(FutureRels),nl,
  %closure([t,4],FutureRels,ClosureRels),
  %write('Rf and tr closure: '),
  %write(ClosureRels),nl,
  equivalent_worlds(FutureRels,Equiv),
  write('Equals worlds: '),
  write(Equiv),nl,
  open(Filename,write,EmptyStream),
  write(EmptyStream,''),
  close(EmptyStream),
  open(Filename,append,Stream),
  print_countermodel_m(Stream,Axioms,Formula,Worlds,FutureRels,Equiv,CoupledList),
  close(Stream).


/* Substitutes each label Y with a pair X-Y, where X is the number associated to the sequent in Count  */
couple_labels([],[]).
couple_labels([(X,Rel,Gamma,Delta)|Seq],[[CRel,CGamma,CDelta]|CList]) :-
  couple_rel(X,Rel,CRel),
  couple_sequent(X,Gamma,CGamma),
  couple_sequent(X,Delta,CDelta),
  couple_labels(Seq,CList).

couple_rel(_,[],[]).
couple_rel(X,[(Y,Z)|Rel],[(X-Y,X-Z)|CRel]) :-
  couple_rel(X,Rel,CRel).

couple_sequent(_,[],[]).
couple_sequent(X,[(Y,A,S)|Seq],[(X-Y,A,S)|Cseq]) :-
  couple_sequent(X,Seq,Cseq).

/* Creates a list of worlds occurring in a coupled sequent */
world_list([],[]).
world_list([[Rel,Gamma,Delta]|CL],FinalW) :-
  world_rel(Rel,WorldsRel),
  world_sequent(Gamma,WorldsGamma),
  world_sequent(Delta,WorldsDelta),
  append(WorldsRel,WorldList,W1),
  append(WorldsGamma,W1,W2),
  append(WorldsDelta,W2,FinalW),
  world_list(CL,WorldList).

world_rel([],[]).
world_rel([(X,Y)|Rel],[X,Y|WorldsRel]) :-
  world_rel(Rel,WorldsRel).

world_sequent([],[]).
world_sequent([(X,_,_)|Sequent],[X|WorldsSeq]) :-
  world_sequent(Sequent,WorldsSeq).

/* Creates a list containg the future relation between coupled labels */
future_rels(_,[],[]).
future_rels(Worlds,[(X,Z)|T],[(X-Y,Z-Y)|NewF]) :-
  member(X-Y,Worlds),
  member(Z-Y,Worlds),
  future_rels(Worlds,T,NewF).
future_rels(Worlds,[_|T],NewF) :-
  future_rels(Worlds,T,NewF).

/* OLD
future_rels(_,_,[],[]).
future_rels(Futures,Worlds,[X-Y|ListWorlds],[(X-Y,Z-Y)|FutureRels]) :-
  member((X,Z),Futures),
  member(Z-Y,Worlds),
  future_rels(Futures,Worlds,ListWorlds,FutureRels).
future_rels(Futures,Worlds,[_|ListWorlds],FutureRels) :-
  future_rels(Futures,Worlds,ListWorlds,FutureRels).
*/

/* Create lists out of all the Rels, Gamma, Delta */
get_rels([],[]).
get_rels([[Rels,_,_]|CoupledList], NewRels) :-
  append(Rels,RelsList,NewRels),
  get_rels(CoupledList,RelsList).

get_gammas([],[]).
get_gammas([[_,Gamma,_]|CoupledList], NewGamma) :-
  append(Gamma,ListGamma,NewGamma),
  get_gammas(CoupledList,ListGamma).

get_deltas([],[]).
get_deltas([[_,_,Delta]|CoupledList], NewDelta) :-
  append(Delta,ListDelta,NewDelta),
  get_deltas(CoupledList,ListDelta).

/* Finds worlds in the reflexive and transitive closure of FutureRels
such that (X-Y, Z-Y) and (Z-Y,X-Y) are both in FutureRels*/
equivalent_worlds([],[]).

equivalent_worlds([(X-Y, Z-Y)|T],[(X-Y, Z-Y)|Equiv]) :-
  %X \= Z,
  member((Z-Y,X-Y),T),
  equivalent_worlds(T,Equiv).

equivalent_worlds([_|T],Equiv) :-
  equivalent_worlds(T,Equiv).

/* Now we can do the actual printing */
print_countermodel_m(Stream,Axioms,Formula,Worlds,FutureRels,Equiv,CoupledList) :-
  write('Intuitionistic Countermodel'),nl,!,
  write(Stream,'\\noindent In detail, the intuitionistic birelational $\\{'),
  tex_axioms(Stream,Axioms),
  write(Stream,'\\}$-countermodel for $'),
  tex_fml(Stream,Formula),
  write(Stream,'$ is as follows: \\par \\medskip \\noindent '),
  write(Stream,'Worlds: \\newline $\\mathcal{W} = \\{'),
  tex_worlds(Stream,Worlds,0),
  write(Stream,'\\}$ \\newline that is all pairs $(i,j)$ such that $j$ is a node in the sequent $i$ in the list above. \\par \\medskip \\noindent '),
  write(Stream,'The future relation is defined by taking the refelxive and transitive closure of the following set: \\newline $\\leq = \\{ '),
  tex_rels(Stream,FutureRels,0,'\\leqslant',8),
  write(Stream,' \\}$ \\newline '),
  write(Stream,'that is all pairs $(i_1,j)\\le(i_2,j)$ with $(i_1,j),(i_2,j)\\in\\mathcal{W}$ and $i_1\\le i_2$ according to the graph above. \\par \\medskip \\noindent '),
  (Equiv\=[],(
  write(Stream,'The following worlds are identified according to the $\\leqslant $ relation:\\newline '),
  tex_rels(Stream,Equiv,0,' = ',8),
  write(Stream,'\\par \\medskip \\noindent '));
  true),
  write(Stream,'Accessibility Relation: \\newline $\\rel = \\{'),
  get_rels(CoupledList,Rels),
  closure(Axioms,Rels,ClosureRels),
  write('Rels :'),
  write(Rels),nl,
  write('Closure Rels :'),
  write(ClosureRels),nl,
  tex_rels(Stream,ClosureRels,0,' R ', 10),
  write(Stream,' \\}$  \\newline '),
  write(Stream,'that is all pairs $(i,j_1)R(i,j_2)$ iff $(i,j_1),(i,j_2)\\in\\mathcal{W}$ and $j_1R j_2$ according to the classical accesibility relation with respect to sequent $i$ in the list above. \\par \\medskip \\noindent '),
  write(Stream,'Valuation Function:\\newline '),
  get_gammas(CoupledList,Gammas),
  get_deltas(CoupledList,Deltas),
  print_valuation_m(Stream,Worlds,Gammas),
  write(Stream,' \\par \\medskip \\noindent Forcing of Formulas: \\newline '),
  print_all_formulas(Stream,Worlds,Gammas,Deltas,' \\Vdash ',' \\nVdash '),
  %print_all_formulas(Stream,Worlds,Gammas,' \\Vdash '),
  %print_all_formulas_order(Stream,Worlds,Gammas,Deltas,' \\Vdash ', ' \\nVdash '),
  %write(Stream,'\\newline'),
  %print_all_formulas(Stream,Worlds,Deltas, ' \\nVdash '),
  write('end countermodel'),nl.

/* print list of worlds */
tex_worlds(_,[],_).
tex_worlds(Stream,[X],_) :-
  tex_pair(Stream,X).
tex_worlds(Stream,[X|Worlds],Counter) :-
  Counter < 20,
  tex_pair(Stream,X),
  write(Stream,', '),
  Counter1 is Counter +1,
  tex_worlds(Stream,Worlds,Counter1).

tex_worlds(Stream,[X|Worlds],Counter) :-
  20 =< Counter,
  tex_pair(Stream,X),
  write(Stream,', $ \\newline $'),
  tex_worlds(Stream,Worlds,0).

/* prints list of future relations
tex_futures(_,[],_).
tex_futures(Stream,[(X,Y)],_) :-
  tex_pair(Stream,X),
  write(Stream,' \\leq '),
  tex_pair(Stream,Y).
tex_futures(Stream,[(X,Y)|FutureRels],Counter) :-
  Counter < 9,
  tex_pair(Stream,X),
  write(Stream,' \\leq '),
  tex_pair(Stream,Y),
  write(Stream,', '),
  Counter1 is Counter +1,
  tex_futures(Stream,FutureRels,Counter1).
tex_futures(Stream,[(X,Y)|FutureRels],Counter) :-
  Counter =< 9,
  tex_pair(Stream,X),
  write(Stream,' \\leq '),
  tex_pair(Stream,Y),
  write(Stream,',  $ \\newline $'),
  tex_futures(Stream,FutureRels,0).
*/

/* prints the list of equals worlds according to <=
tex_equiv(_,[],_).
tex_equiv(Stream,[(X,Y)],_) :-
  tex_pair(Stream,X),
  write(Stream,' = '),
  tex_pair(Stream,Y).
tex_equiv(Stream,[(X,Y)|Equiv],Counter) :-
  Counter < 9,
  tex_pair(Stream,X),
  write(Stream,' = '),
  tex_pair(Stream,Y),
  write(Stream,', '),
  Counter1 is Counter +1,
  tex_futures(Stream,FutureRels,Counter1).
tex_futures(Stream,[(X,Y)|FutureRels],Counter) :-
  Counter =< 9,
  tex_pair(Stream,X),
  write(Stream,' \\leq '),
  tex_pair(Stream,Y),
  write(Stream,',  $ \\newline $'),
  tex_futures(Stream,FutureRels,0).
*/

/* prints list of relations (future, rels) */
tex_rels(_,[],_,_,_).
tex_rels(Stream,[(X,Y)|[]],_,Symbol,_) :-
  tex_pair(Stream,X),
  write(Stream,Symbol),
  tex_pair(Stream,Y).
tex_rels(Stream,[(X,Y)|FutureRels],Counter,Symbol,Space) :-
  Counter < Space,
  tex_pair(Stream,X),
  write(Stream,Symbol),
  tex_pair(Stream,Y),
  write(Stream,', '),
  Counter1 is Counter +1,
  tex_rels(Stream,FutureRels,Counter1,Symbol,Space).
tex_rels(Stream,[(X,Y)|FutureRels],Counter,Symbol,Space) :-
  Space =< Counter,
  tex_pair(Stream,X),
  write(Stream,Symbol),
  tex_pair(Stream,Y),
  write(Stream,', $ \\newline $'),
  tex_rels(Stream,FutureRels,0,Symbol,Space).

/* prints the valuation: atomic formulas in all antecedents */

print_valuation_m(_,[],_).
print_valuation_m(Stream,[X-Y|Worlds],Gamma) :-
  occurs_atom(X-Y,Gamma),
  write(Stream,'$ \\llbracket'),
  tex_pair(Stream,X-Y),
  write(Stream,'\\rrbracket = \\{'),
  print_atoms(Stream,X-Y,Gamma),
  write(Stream,' \\}$'),
  write(Stream,'\\newline'),
  print_valuation_m(Stream,Worlds,Gamma).

print_valuation_m(Stream,[X-Y|Worlds],Gamma) :-
  \+occurs_atom(X-Y,Gamma),
  write(Stream, '$ \\llbracket'),
  tex_pair(Stream,X-Y),
  write(Stream,' \\rrbracket = \\varnothing $'),
  write(Stream,'\\newline'),
  print_valuation_m(Stream,Worlds,Gamma).

occurs_atom(_,[]) :- false.
occurs_atom(X,[(X,A,_)|_]) :- atom(A).
occurs_atom(X,[_|T]) :- occurs_atom(X,T).

print_atoms(_,_,[]).
print_atoms(Stream,X,[(X,A,_)|T]) :-
  atom(A),
  write(Stream,A),
  (occurs_atom(X,T), write(Stream,', ');write(Stream,'') ),
  print_atoms(Stream,X,T).
print_atoms(Stream,X,[_|T]) :-
  print_atoms(Stream,X,T).

/* prints all non-atomic formulas in gamma */
/*
print_all_formulas_order(_,[],_,_,_).
print_all_formulas_order(Stream,[X|Worlds],Sforce,Sunforce,ForceSymbol,UnforceSymbol):-
  %%member((X,_,_),Sforce),
  print_formulas(Stream,X,Sforce,ForceSymbol),
  write(Stream,'\\newline'),
  print_formulas(Stream,X,Sunforce,UnforceSymbol),
  write(Stream,'\\newline'),
  print_all_formulas_order(Stream,Worlds,Sforce,Sunforce,ForceSymbol,UnforceSymbol).
*/

print_all_formulas(_,[],_,_,_,_).
print_all_formulas(Stream,[X|Worlds],Gamma,Delta,Force,NotForce):-
  (member((X,_,_),Gamma),(print_formulas(Stream,X,Gamma,Force),write(Stream,'\\newline')) ;true),
  (member((X,_,_),Delta),(print_formulas(Stream,X,Delta,NotForce),write(Stream,'\\newline')) ;true),
  print_all_formulas(Stream,Worlds,Gamma,Delta,Force,NotForce).




print_all_formulas(_,_,[],_).
print_all_formulas(_,[],_,_).
print_all_formulas(Stream,[X|Worlds],Sequent,Symbol):-
  (member((X,_,_),Sequent),
  (print_formulas(Stream,X,Sequent,Symbol),
  write(Stream,'\\newline'),
  print_all_formulas(Stream,Worlds,Sequent,Symbol));
  print_all_formulas(Stream,Worlds,Sequent,Symbol)).


print_formulas(_,_,[],_).
print_formulas(Stream,X-Y,[(X-Y,A,_)|T],Symbol) :-
  %%\+atom(A),
  write(Stream,'$ '),
  tex_pair(Stream,X-Y),
  write(Stream, Symbol),
  tex_fml(Stream,A),
  write(Stream,' $'),
  (member((X-Y,_,_),T),write(Stream,', ') ;write(Stream,'') ),
  print_formulas(Stream,X-Y,T,Symbol).
print_formulas(Stream,X,[_|T],Symbol) :-
  print_formulas(Stream,X,T,Symbol).

/* prints a world (pair)*/

tex_pair(Stream,X-Y) :-
  write(Stream,'('),
  write(Stream,X),write(Stream,','),write(Stream,Y),
  write(Stream,')').
tex_pair(Stream,X) :- write(Stream,X).



/* LaTeX a formula */
tex_fml(Stream,false) :- write(Stream,'\\bot ').
tex_fml(Stream,true) :- write(Stream,'\\top ').
tex_fml(Stream,A) :- atom(A), write(Stream,A).
tex_fml(Stream,!A) :- write(Stream,' \\Box '),tex_fml(Stream,A).
%%tex_fml(Stream,!A) :- (ismodal(A);isliteral(A)),write(Stream,' \\Box '),tex_fml(Stream,A).
%%tex_fml(Stream,!A) :- write(Stream,' \\Box '),write(Stream,'('),tex_fml(Stream,A),write(Stream,')').
tex_fml(Stream,?A) :- write(Stream,' \\Diamond '),tex_fml(Stream,A).
%%tex_fml(Stream,?A) :- (ismodal(A);isliteral(A)),write(Stream,' \\Diamond '),tex_fml(Stream,A).
%%tex_fml(Stream,?A) :- write(Stream,' \\Diamond '),write(Stream,'('),tex_fml(Stream,A),write(Stream,')').
tex_fml(Stream,~A) :- atom(A),write(Stream, '\\bar{'), write(Stream,A), write(Stream,'}').
tex_fml(Stream,~(A)) :- write(Stream, '\\overline{'), tex_fml(Stream,A), write(Stream,'}').
tex_fml(Stream,A^B) :- write(Stream,'('),tex_fml(Stream,A),write(Stream,'\\vlan '),tex_fml(Stream,B),write(Stream,')').
tex_fml(Stream,A v B) :- write(Stream,'('),tex_fml(Stream,A),write(Stream,'\\vlor '),tex_fml(Stream,B),write(Stream,')').
tex_fml(Stream,A->B) :- write(Stream,'('),tex_fml(Stream,A),write(Stream,'\\vljm '),tex_fml(Stream,B),write(Stream,')').
%tex_fml(_,A) :- write(A), write(' -???- ').

negatom(~A) :- atom(A).
ismodal(!_).
ismodal(?_).
isliteral(A) :- (atom(A);negatom(A)).

/* LaTeX a rule name */
/* classical modal logic */
tex_rule(Stream,init) :- write(Stream,'\\axId').
tex_rule(Stream,top) :- write(Stream,'\\axTop').
tex_rule(Stream,andK) :- write(Stream,'\\Kand').
tex_rule(Stream,orK) :- write(Stream,'\\Kor').
tex_rule(Stream,boxK) :- write(Stream,'\\Kbox').
tex_rule(Stream,diamondK) :- write(Stream,'\\Kdiamond').
tex_rule(Stream,diamondKD) :- write(Stream,'\\Kd').
tex_rule(Stream,diamondKT) :- write(Stream,'\\Kt').
tex_rule(Stream,diamondKB) :- write(Stream,'\\Kb').
tex_rule(Stream,diamondK4) :- write(Stream,'\\Kfour').
tex_rule(Stream,diamondK5) :- write(Stream,'\\Kfive').
tex_rule(Stream,fail) :- write(Stream,'\\fail').

/* single conclusion & multi conclusion  */
tex_rule(Stream,false) :- write(Stream,'\\axBot').
tex_rule(Stream,andL) :- write(Stream,'\\andL').
tex_rule(Stream,andR_2p) :- write(Stream,'\\andR').
tex_rule(Stream,orL_2p) :- write(Stream,'\\orL').
tex_rule(Stream,diamondL) :- write(Stream,'\\diamondL').
tex_rule(Stream,diamondR) :- write(Stream,'\\diamondR').
tex_rule(Stream,boxL) :- write(Stream,'\\boxL').

tex_rule(Stream,boxD) :- write(Stream,'\\boxD').
tex_rule(Stream,boxT) :- write(Stream,'\\boxT').
tex_rule(Stream,boxB) :- write(Stream,'\\boxB').
tex_rule(Stream,box4) :- write(Stream,'\\boxFour').
tex_rule(Stream,box5) :- write(Stream,'\\boxFive').

/* only single conclusion  */
tex_rule(Stream,orR1s) :- write(Stream,'\\orRs{1}').
tex_rule(Stream,orR2s) :- write(Stream,'\\orRs{2}').
tex_rule(Stream,implRs) :- write(Stream,'\\implRs').
tex_rule(Stream,implLs) :- write(Stream,'\\implLs').
tex_rule(Stream,boxRs) :- write(Stream,'\\boxRs').

tex_rule(Stream,diamondDs) :- write(Stream,'\\diamondDs').
tex_rule(Stream,diamondTs) :- write(Stream,'\\diamondTs').
tex_rule(Stream,diamondBs) :- write(Stream,'\\diamondBs').
tex_rule(Stream,diamond5s) :- write(Stream,'\\diamondFives').
tex_rule(Stream,diamond4s) :- write(Stream,'\\diamondFours').

tex_rule(Stream,bracketsD) :- write(Stream,'\\bracketsD').

/* only multi conclusion  */
tex_rule(Stream,orRm) :- write(Stream,'\\orRm').
tex_rule(Stream,implLm) :- write(Stream,'\\implLm').
tex_rule(Stream,implRm) :- write(Stream,'\\implRm').
tex_rule(Stream,boxRm) :- write(Stream,'\\boxRm').

tex_rule(Stream,diamondDm) :- write(Stream,'\\diamondDm').
tex_rule(Stream,diamondTm) :- write(Stream,'\\diamondTm').
tex_rule(Stream,diamondBm) :- write(Stream,'\\diamondBm').
tex_rule(Stream,diamond5m) :- write(Stream,'\\diamondFivem').
tex_rule(Stream,diamond4m) :- write(Stream,'\\diamondFourm').

tex_system(Stream,k) :- write(Stream,'\\sysK').
tex_system(Stream,i) :- write(Stream,'\\sysI').
tex_system(Stream,m) :- write(Stream,'\\sysM').


/* Latex names of axioms  */
tex_axioms(_,[]).
tex_axioms(Stream,[d1]) :- write(Stream,'\\axDbra').
tex_axioms(Stream,[d]) :- write(Stream,'\\axD').
tex_axioms(Stream,[d|T]) :- write(Stream,'\\axD'), write(Stream,','),tex_axioms(Stream,T).
tex_axioms(Stream,[t]) :- write(Stream,'\\axT').
tex_axioms(Stream,[t|T]) :- write(Stream,'\\axT'), write(Stream,','), tex_axioms(Stream,T).
tex_axioms(Stream,[b]) :- write(Stream,'\\axB').
tex_axioms(Stream,[b|T]) :- write(Stream,'\\axB'), write(Stream,','), tex_axioms(Stream,T).
tex_axioms(Stream,[4]) :- write(Stream,'\\axFour').
tex_axioms(Stream,[4|T]) :- write(Stream,'\\axFour'), write(Stream,','), tex_axioms(Stream,T).
tex_axioms(Stream,[5]) :- write(Stream,'\\axFive').
tex_axioms(Stream,[5|T]) :- write(Stream,'\\axFive'), write(Stream,','), tex_axioms(Stream,T).

/*
tex_axioms(Stream,[k|T]) :- tex_axioms(Stream,T).
tex_axioms(Stream,[i|T]) :- tex_axioms(Stream,T).
tex_axioms(Stream,[m|T]) :- tex_axioms(Stream,T).
 */

/*********************** Print sequents in the terminal *******************/

ter_seq_k([Rel,Lambda],N) :-
      ter_input(Lambda,N,0),
      ter_nesting_k(Rel,N,[Rel,Lambda]),!.

ter_seq_i([Rel,Lambda,Out],N) :-
      ter_input(Lambda,N,0),
      ter_output(Out,N,Lambda),
      ter_nesting_i(Rel,N,[Rel,Lambda,Out]),!.

ter_seq_m([Rel,Gamma,Delta],N) :-
      ter_input(Gamma,N,0),
      ter_output_m(Delta,N,0,Gamma),
      ter_nesting_m(Rel,N,[Rel,Gamma,Delta]),!.

ter_input([],_,_).
ter_input([(N,F,+)|T],N,Count) :-
      write(F),
      ( (T = [(N,_,+)|_]), write(','); write('') ),
      NewCount is Count+1,
      ter_input(T,N,NewCount).
ter_input([(N,_,-)|T],N,Count) :-
      ( (Count \= 0, T = [(N,_,+)|_]), write(','); write('') ),
      ter_input(T,N,Count).
ter_input([(_,_,_)|T],N,Count) :-
      ( (Count \= 0, T = [(N,_,+)|_]), write(','); write('') ),
      ter_input(T,N,Count).

ter_output((N,F,_),N,Lambda) :-
      ( (member((N,_,+),Lambda)),write(','); write('') ),
      write(F),
      write('°').
ter_output((_,_),_,_).


ter_output_m([],_,_,_).
ter_output_m([(N,F,+)|T],N,Count,Gamma) :-
  ( (member((N,_,+),Gamma), Count = 0),write(', '); write('') ),
  write(F),
  write('°'),
  ( (T = [(N,_,+)|_]), write(', '); write('') ),
  NewCount is Count+1,
  ter_output_m(T,N,NewCount,Gamma).
ter_output_m([(N,_,-)|T],N,Count,Gamma) :-
  ( (Count \= 0, T = [(N,_,+)|_]), write(', '); write('') ),
  ter_output_m(T,N,Count,Gamma).
ter_output_m([(_,_,_)|T],N,Count,Gamma) :-
  ( (Count \= 0, T = [(N,_,+)|_]), write(', '); write('') ),
  ter_output_m(T,N,Count,Gamma).


ter_nesting_k([],_,_).
ter_nesting_k([(N,M)|T],N,Seq) :-
      Seq = [_,Lambda],
      (member((N,_,+),Lambda), write(',') ; write('')),
      write('['),
      ter_seq_k(Seq,M),
      write(']'),
      ter_nesting_k(T,N,Seq).
ter_nesting_k([(_,_)|T],N,Seq) :-
      ter_nesting_k(T,N,Seq).

ter_nesting_i([],_,_).
ter_nesting_i([(N,M)|T],N,Seq) :-
      Seq = [_,Lambda,(X,_)],
      ((member((N,_,+),Lambda);(X=N)), write(',') ; write('')),
      write('['),
      ter_seq_i(Seq,M),
      write(']'),
      ter_nesting_i(T,N,Seq).
ter_nesting_i([(_,_)|T],N,Seq) :-
      ter_nesting_i(T,N,Seq).

ter_nesting_m([],_,_).
ter_nesting_m([(N,M)|T],N,Seq) :-
      Seq = [_,Gamma,Delta],
      ((member((N,_,+),Gamma);member((N,_,+),Delta)), write(',') ; write('')),
      write('['),
      ter_seq_m(Seq,M),
      write(']'),
      ter_nesting_m(T,N,Seq).
ter_nesting_m([(_,_)|T],N,Seq) :-
      ter_nesting_m(T,N,Seq).


/* tex_axioms(Stream,[m|T]) :- member(k,T),tex_axioms(Stream,T).
tex_axioms(Stream,[m|T]) :- write(Stream,'\\axM'),tex_axioms(Stream,T).
tex_axioms(Stream,[k|T]) :- write(Stream,'\\axK'), tex_axioms(Stream,T).
tex_axioms(Stream,[i|T]) :- write(Stream,'\\axI'), tex_axioms(Stream,T).*/

/*
Derivable intuitionistic formulas:

derive(i,[], false->a).
derive(i,[], ((a v b)->false)-> ((a->false)^(b->false)) ).
derive(i,[], ((a->false)^(b->false))-> ((a v b)->false) ).
derive(i,[], ((a->b)->false)-> ((a->false)v(b->false))).
derive(i,[], ((a->b)->false)->(((a->false)^ b)->false)  ).

Derivable intuitionistic modal formulas:

k1: derive(i,[], (!(a->b)) -> ((!a)->(!b)) ).
k2: derive(i,[], (!(a->b)) -> ((?a)->(?b)) ).
k3: derive(i,[], (?(a v b)) -> ((?a)v(?b))).
k4: derive(i,[], ((?a)-> (!b)) -> (!(a->b))).
k5: derive(i,[], (?false)->false).

derive(i,[], ( (?(a v b)) ^ (!(?(a v b))) ) -> ( (?a)v(?b) ) ).

Extensions (intuitionistic)

d: derive(i,[d],(!a)->(?a)).
t: derive(i,[t],(a->(?a))^((!a)->a)).
b: derive(i,[b], (a->(!(?a)))^((?(!a)))->a).
4: derive(i,[4], ((?(?a))->(?a))^((!a)->(!(!a))) ).
5: derive(i,[5], (?a->(!(?a))) ^ ( (?(!a))->!a ) ).

Extensions (classic)
d: derive(k,[d], !a -> ?a).
t: derive(k,[t], a -> ?a).
   derive(k,[t], !a -> a).
b: derive(k,[b], a -> ! (?a)).
   derive(k,[b], ? (!a)-> a).
4: derive(k,[4],!a -> !(!a)).
   derive(k,[4],?(?a)-> ?a).
5: derive(k,[5], ?a -> !(?a)).

NOT derivable intuitionistic formulas:
derive(i,[],a v (a->false)).
derive(i,[],(a->b)v(b->a)).
derive(i,[],((a->false)->false)->a).
derive(i,[],(((a->false)^(b->false))->false)->(a v b)).
derive(i,[],(((a->false)v(b->false))->false)->(a^ b)).


From the command line:
swipl -q -f prover_modal_nested.pl -g "derive([],a->a)" -t halt


Formula for which the repetition of the implication in the left premiss of L-> is needed
derive(i,[],(( a v (a->false))-> false) -> false).

Looping formulas
derive(i,[],(( a v (a->false))-> false) -> false).
derive(i,[d],((a->(?b))->a)->a).
derive(i,[t,b],!(?((a->(?a))->a )->a ) ).
derive(k,[4 ], ( (? (! a) v (!(?(! a )))))).

*/
