/*******************************
 Name: mrmsim.cxx
 Author: Leo Liberti
 Purpose: Marvin Minsky's register machine simulator
 Source: GNU C++
 History: 130221 work started
********************************/

#include <iostream>
#include <set>
#include <map>
#include <vector>
#include <fstream>
#include <sstream>
#include <cstring>

struct instruction {
  int index;
  int type;
  int goto1;
  int goto2;
};

int main(int argc, char** argv) {
  using namespace std;
  // the registers: indexing starts from 1
  vector<int> r(1);
  vector<int> rprev(1);
  // the program
  map<int,instruction> program;

  if (argc < 2) {
    cerr << argv[0] << ": Usage: " << argv[0] << " [-d] program.mrm" << endl;
    cerr << "  -d yields verbose output for debugging" << endl;
    cerr << "  program.mrm is an ASCII-128 file describing a MRM program"
	 << endl;
    cerr << " MRM program syntax:" << endl;
    cerr << "  a MRM program is a list of initial register values followed by"
	 << endl;
    cerr << "    a list of instructions" << endl;
    cerr << "  a list of initial register values is a line starting with 'R'"
	 << endl;
    cerr << "    and followed by as many values as there are registers"
	 << endl;
    cerr << "  an instruction is a quintuplet (j,b,h,l) of ints in the form:" 
	 << endl;
    cerr 
      << "    instruction_number : register_index instruction_type goto1 goto2"
      << endl;
    cerr << "  instruction_number is a label for 'goto' statements to refer to"
	 << endl;
    cerr << "    0 is the halting instruction, -1 is the error instruction"
	 << endl;
    cerr << "  register_index is the register index (starts from 1)" << endl;
    cerr << "  instruction_type can be 0 (increase) or 1 (decrease)" << endl;
    cerr << "  goto1, goto2 are two existing instruction numbers" << endl;
    cerr << "  instructions beginning with # are comments" << endl;
    cerr << " MRM program semantics:" << endl;
    cerr << "  (j,0,h,l) = 'reg[j]++; goto h;'" << endl;
    cerr << "  (j,1,h,l) = 'if (reg[j] == 0) goto l; else {reg[j]--; goto h}'"
	 << endl;
    exit(1);
  }
  
  int argcounter = 1;
  bool debug = false;
  if (strncmp(argv[argcounter], "-d", 2) == 0) {
    debug = true;
    argcounter++;
  }

  ifstream prog(argv[argcounter]);
  if (!prog) {
    cerr << argv[0] << ": can't open " << argv[argcounter] 
	 << " for reading" << endl;
    exit(2);
  }

  char c;
  char prevc;
  char nextc;
  int lineno = 0;
  while(!prog.eof()) {
    // read a char off the input file
    prog.get(c);
    if (c != ' ' && c != '#') {
      // skip all spaces and comment chars
      stringstream instr;
      bool registerlineflag = false;
      bool emptylineflag = false;
      while (c != '\n' && !prog.eof()) {
	// copy rest of (good) line to instr
	if (c == ':') {
	  // ':' syntax only for user, turn to space
	  c = ' ';
	} else if (c == ' ' || c == '\t') {
	  prog.get(nextc);
	  prog.unget();
          if (nextc != '\n' && prevc != ' ' && prevc != '\t') {
	    // convert tab to space, only if not followed by EOL
	    instr << ' ';
	  } else {
	    // just store one space character per sequence of spaces/tabs
	  }
	} else if (c == 'R') {
	  // register value line
	  registerlineflag = true;
	} else {
	  instr << c;
	}
	prevc = c;
	prog.get(c);
      }
      if (c == '\n') {
	// line counter
	lineno++;
	if (instr.str().size() == 0) {
	  // empty line
	  emptylineflag = true;
	} else {
	  emptylineflag = false;
	}
      }
      // deal with empty lines
      if (registerlineflag && !emptylineflag) {
	// parse initial values for registers
	int regval = 0;
	while(!instr.eof()) {
	  instr >> regval;
	  r.push_back(regval);
	}
	registerlineflag = false;
      } else if (!emptylineflag) {
	// parse the instruction
	int label;
	instruction I;
	instr >> label >> I.index >> I.type >> I.goto1 >> I.goto2;
	// verify instruction syntax
	if (label < -1) {
	  cerr << argv[0] << ": error at line " << lineno 
	       << ": labels range in [-1,+infty]" << endl;
	  exit(3);
	}
	if (I.index < 0 || I.index >= r.size()) {
	  cerr << argv[0] << ": error at instruction " << label 
	       << ": regindex value " << I.index 
	       << " not in [0," << r.size()-1 << "]"
	       << endl;
	  exit(4);
	}
	if (I.type < 0 || I.type > 1) {
	  cerr << argv[0] << ": error at instruction " << label
	       << ": instruction types range in {0,1}" << endl;
	  exit(5);
	} 
	if (I.goto1 < -1) {
	  cerr << argv[0] << ": error at instruction " << label
	       << ": labels range in [-1,+infty]" << endl;
	  exit(6);
	}
	if (I.type == 1 && I.goto2 < -1) {
	  cerr << argv[0] << ": error at instruction " << label
	       << ": labels range in [-1,+infty]" << endl;
	  exit(7);
	}
	// store instruction
	if (label != 0) {
	  // ignore HALT instruction, it's standard anyway
	  program[label] = I;
	}
      } else {
	// empty line, do nothing
      }
    } else if (c == '#') {
      // comment lines
      while (c != '\n' && !prog.eof()) {
	// advance to end of line
	prog.get(c);
      }
      // increase line counter
      lineno++;
    }
  }
  prog.close();
  
  if (debug) {
    // verify - print out program
    cout << "## register input" << endl;
    cout << "R";
    for(int i = 1; i < r.size(); i++) {
      cout << " " << r[i];
    }
    cout << endl;
    cout << "## program" << endl;
    cout << "# 0: HALT" << endl;
    for(map<int,instruction>::iterator mi = program.begin(); 
	mi != program.end(); mi++) {
      instruction& I = mi->second;
      cout << mi->first << ": " << I.index << " " << I.type 
	   << " " << I.goto1 << " " << I.goto2 << endl;
    }
    cout << endl;
  }
  
  if (debug) {
    cout << "## simulation with input";
  }
  for(int i = 1; i < r.size(); i++) {
    cout << " " << r[i];
  }
  cout << endl;

  // simulate MRM
  int current = 1;
  int previous = 1;
  
  // keep running until HALT (0) or ERROR (-1)
  while(current != 0) {

    if (current == -1) {
      // error
      cout << "[error] from instruction " << previous << endl;
      break;
    }

    // fetch current instruction
    instruction& I = program[current];

    if (debug) {
      // save registers before carrying out current instructions
      for(int i = 1; i < r.size(); i++) {
	rprev[i] = r[i];
      }
    }

    // semantics
    if (I.type == 0) {
      r[I.index]++;
      current = I.goto1;
    } else {
      if (r[I.index] == 0) {
	current = I.goto2;
      } else {
	r[I.index]--;
	current = I.goto1;
      }
    }    

    if (debug) {
      cout << "[" << previous << ":" << I.index << "," << I.type 
	   << "," << I.goto1 << "," << I.goto2 << "] (";
      for(int i = 1; i < r.size(); i++) {
	cout << " " << rprev[i];
      }
      cout << " ) =";
    }

    // print current register values
    for(int i = 1; i < r.size(); i++) {
      cout << " " << r[i];
    }
    if (debug) {
      cout << "  # ";
      if (I.type == 0) {
	cout << "R[" << I.index << "]++; goto " << I.goto1 << ";";
      } else {
	if (rprev[I.index] == 0) {
	  cout << "goto " << I.goto2 << ";";
	} else {
	  cout << "R[" << I.index << "]--; goto " << I.goto1 << ";";
	}
      }
    }
    cout << endl;

    // save previous label
    previous = current;
  }

  return 0;
}
