# marvin minsky's register machine

## sets
# set of register indices
param mm integer, >= 0;
set M := 1..mm;

# set of register values
# max value for registers
param vm integer, >= 1;
set V := 1..vm+1;

# set of states
param nm integer, >= 0;
set N := 1..nm;
set N0 := 0..nm;

# set of time instants
param tm integer, >= 1;
set T := 1..tm;
set T0 := 0..tm;

## parameters
# register index (1st component of instruction i in N)
param h{N} integer, >= 0, <= mm;

# instruction type (2nd component of instruction i in N)
param b{N} binary;

# target state (3rd component of instruction i in N)
param k{N} integer, >= 0, <= nm;

# target state 2 (4th component of instruction i in N)
param l{N} integer, >= 0, <= nm;

# starting state
param s0 integer, >= 0, default 1;

# lower/upper bounds for register variables (from 0)
param rL{M} default 0;
param rU{M} default vm;
param r0L{M} default 0;
param r0U{M} default vm;

## decision variables
# value in register at timestep
var r{j in M,T0} integer, >= rL[j]+1, <= rU[j]+1;
# binary representation
var rb{M,T0,V} binary;

# find most difficult instance: decide registers (the "instance") so that 
# time to termination is maximized
var r0{j in M} integer, >= max(r0L[j]+1,rL[j]+1), <= min(r0U[j]+1,rU[j]+1);

# 1 if value in register is positive
var rho{M,T0}, binary;

# current status index at timestep
var x{N0,T0} binary;

## linearization variables
# linearization of rbx
var rx{N0,M,T,V} >= 0, <= 1;
# linearization of rbx_t-1
var rxm{N0,M,T,V} >= 0, <= 1;
# linearization of rhox
var rhox{N,T} >= 0, <= 1;
# linearization of xx_k
var xxk{N,T} >= 0, <= 1;
# linearization of xx_l
var xxl{N,T} >= 0, <= 1;
# linearization of xrhox
var xrhox{N,T} >= 0, <= 1;
# linearization of rrhox
var rrhox{N,T,V} >= 0, <= 1;
# linearization of r_t-1rhox
var rmrhox{N,T,V} >= 0, <= 1;
# linearization of xkrhox
var xkrhox{N,T} >= 0, <= 1;
# linearization of xx_t-1
var xx0{T} >= 0, <= 1;

## objective

minimize idle_time: sum{t in T} x[0,t];

## constraints

## algorithm semantic
# starting values
subject to init_register{j in M} : r[j,0] = r0[j];
subject to init_state : x[s0,0] = 1;

# if b=0, Rj++
subject to incr{t in T, i in N} : 
  sum{v in V} (1-b[i])*v*rx[i,h[i],t,v] = 
  sum{v in V} (1-b[i])*v*rxm[i,h[i],t,v] + (1-b[i])*x[i,t-1];

# if b=0, c <-- k
subject to setc_k{t in T, i in N} : 
  (1-b[i])*xxk[i,t] = (1-b[i])*x[i,t-1];

# if b=1 and Rj>0, decrement Rj
subject to decr{t in T, i in N} :
  sum{v in V} b[i]*v*rrhox[i,t,v] =
  sum{v in V} b[i]*v*rmrhox[i,t,v] - b[i]*rhox[i,t];

# if b=1 and Rj>0, c <-- k
subject to bsetc_k{t in T, i in N} : 
  b[i]*xkrhox[i,t] = b[i]*rhox[i,t];

# if b=1 and Rj=0, fix Rh
subject to fixRh{t in T, i in N} : 
  sum{v in V} b[i]*v*(rx[i,h[i],t,v] - rrhox[i,t,v]) = 
  sum{v in V} b[i]*v*(rxm[i,h[i],t,v] - rmrhox[i,t,v]);

# if b=1 and Rj=0, c <-- l
subject to setc_l{t in T, i in N} : 
  b[i]*(xxl[i,t] - xrhox[i,t]) = b[i]*(x[i,t-1] - rhox[i,t]);

# if c=i and j!=h[i], fix R_j
subject to fixRj{t in T, i in N, j in M diff {h[i]}} :
  sum{v in V} v*rx[i,j,t,v] = sum{v in V} v*rxm[i,j,t,v];

# if c=0 stop
subject to stop{t in T} : xx0[t] = x[0,t-1];

# if c=0 then fix Rj's
subject to fixRj0{t in T, j in M} : 
  sum{v in V} v*rx[0,j,t,v] = sum{v in V} rxm[0,j,t,v];

## definitions
# c takes one value at any time
subject to xass{t in T0} : sum{i in N0} x[i,t] = 1;

# def of rho in terms of r
subject to rhodef1{j in M, t in T0} : r[j,t] - 1 >= rho[j,t];
subject to rhodef2{j in M, t in T0} : r[j,t] - 1 <= (vm + 1)*rho[j,t];

## reformulations
# definition of rb - Int2Bin
subject to rbdef{j in M, t in T0} : sum{v in V} v*rb[j,t,v] = r[j,t];
subject to rbass{j in M, t in T0} : sum{v in V} rb[j,t,v] = 1;

# linearization of rx - ProdBin
subject to rxlin1{i in N, j in M, t in T, v in V} : rx[i,j,t,v] <= rb[j,t,v];
subject to rxlin2{i in N, j in M, t in T, v in V} : rx[i,j,t,v] <= x[i,t-1];
subject to rxlin3{i in N, j in M, t in T, v in V} : 
  rx[i,j,t,v] >= rb[j,t,v]+x[i,t-1]-1;

# linearization of rxm
subject to rxmlin1{i in N, j in M, t in T, v in V} : 
  rxm[i,j,t,v] <= rb[j,t-1,v];
subject to rxmlin2{i in N, j in M, t in T, v in V} : rxm[i,j,t,v] <= x[i,t-1];
subject to rxmlin3{i in N, j in M, t in T, v in V} : rxm[i,j,t,v] >= 
  rb[j,t-1,v] + x[i,t-1] - 1;

# linearization of xx_k
subject to xxklin1{i in N, t in T} : xxk[i,t] <= x[k[i],t];
subject to xxklin2{i in N, t in T} : xxk[i,t] <= x[i,t-1];
subject to xxklin3{i in N, t in T} : xxk[i,t] >= x[k[i],t] + x[i,t-1] - 1;

# linearization of rhox
subject to rhoxlin1{i in N, t in T} : rhox[i,t] <= rho[h[i],t-1];
subject to rhoxlin2{i in N, t in T} : rhox[i,t] <= x[i,t-1];
subject to rhoxlin3{i in N, t in T} : rhox[i,t] >= 
  rho[h[i],t-1] + x[i,t-1] - 1;

# linearization of rrhox
subject to rrhoxlin1{i in N, t in T, v in V} : rrhox[i,t,v] <= rb[h[i],t,v];
subject to rrhoxlin2{i in N, t in T, v in V} : rrhox[i,t,v] <= rhox[i,t];
subject to rrhoxlin3{i in N, t in T, v in V} : rrhox[i,t,v] >= 
  rb[h[i],t,v] + rhox[i,t] - 1;

# linearization of rmrhox
subject to rmrhoxlin1{i in N, t in T, v in V} : 
  rmrhox[i,t,v] <= rb[h[i],t-1,v];
subject to rmrhoxlin2{i in N, t in T, v in V} : rmrhox[i,t,v] <= rhox[i,t];
subject to rmrhoxlin3{i in N, t in T, v in V} : rmrhox[i,t,v] >= 
  rb[h[i],t-1,v] + rhox[i,t] - 1;

# linearization of xkrhox
subject to xkrhoxlin1{i in N, t in T} : xkrhox[i,t] <= x[k[i],t];
subject to xkrhoxlin2{i in N, t in T} : xkrhox[i,t] <= rhox[i,t];
subject to xkrhoxlin3{i in N, t in T} : 
  xkrhox[i,t] >= x[k[i],t] + rhox[i,t] - 1;

# linearization of xx_l
subject to xxllin1{i in N, t in T} : xxl[i,t] <= x[l[i],t];
subject to xxllin2{i in N, t in T} : xxl[i,t] <= x[i,t-1];
subject to xxllin3{i in N, t in T} : xxl[i,t] >= x[l[i],t] + x[i,t-1] - 1;

# linearization of xrhox
subject to xrhoxlin1{i in N, t in T} : xrhox[i,t] <= x[l[i],t];
subject to xrhoxlin2{i in N, t in T} : xrhox[i,t] <= rhox[i,t];
subject to xrhoxlin3{i in N, t in T} : xrhox[i,t] >= x[l[i],t] + rhox[i,t] - 1;

# linearization of xx0
subject to xx0lin1{t in T} : xx0[t] <= x[0,t];
subject to xx0lin2{t in T} : xx0[t] <= x[0,t-1];
subject to xx0lin3{t in T} : xx0[t] >= x[0,t] + x[0,t-1] - 1;

