# marvin minsky's register machine

## sets
# set of register indices
param mm integer, >= 0;
set M := 1..mm;

# set of register values
# max value for registers
param vm integer, >= 1;

# set of states
param nm integer, >= 0;
set N := 1..nm;
set N0 := 0..nm;

# set of time instants
param tm integer, >= 1;
set T := 1..tm;
set T0 := 0..tm;

## parameters
# register index (1st component of instruction i in N)
param h{N} integer, >= 0, <= mm;

# instruction type (2nd component of instruction i in N)
param b{N} binary;

# target state (3rd component of instruction i in N)
param k{N} integer, >= 0, <= nm;

# target state 2 (4th component of instruction i in N)
param l{N} integer, >= 0, <= nm;

# starting values in registers (the "instance")
param r0{M} integer, >= 0, <= vm;

# starting state
param s0 integer, >= 0, default 1;

## decision variables
# value in register at timestep
var r{M,T0} integer, >= 1, <= vm+1;

# 1 if value in register is positive
var rho{M,T0}, binary;

# current status index at timestep
var x{N0,T0} binary;

## objective

minimize nothing: 0;

## constraints

## algorithm semantic
# starting values
subject to init_register{j in M} : r[j,0] = r0[j] + 1;
subject to init_state : x[s0,0] = 1;

# if b=0, Rj++
subject to incr{t in T, i in N} : 
  (1-b[i])*r[h[i],t]*x[i,t-1] = (1-b[i])*(r[h[i],t-1] + 1)*x[i,t-1];

# if b=0, c <-- k
subject to setc_k{t in T, i in N} : 
  (1-b[i])*x[k[i],t]*x[i,t-1] = (1-b[i])*x[i,t-1];

# if b=1 and Rj>0, decrement Rj
subject to decr{t in T, i in N} :
  b[i]*r[h[i],t]*rho[h[i],t-1]*x[i,t-1] =
  b[i]*(r[h[i],t-1] - 1)*rho[h[i],t-1]*x[i,t-1];

# if b=1 and Rj>0, c <-- k
subject to bsetc_k{t in T, i in N} : 
  b[i]*x[k[i],t]*rho[h[i],t-1]*x[i,t-1] = b[i]*rho[h[i],t-1]*x[i,t-1];

# if b=1 and Rj=0, fix Rj
subject to fixRh{t in T, i in N} :
  b[i]*(1-rho[h[i],t-1])*x[i,t-1]*r[h[i],t] = 
  b[i]*(1-rho[h[i],t-1])*x[i,t-1]*r[h[i],t-1];

# if b=1 and Rj=0, c <-- l
subject to setc_l{t in T, i in N} : 
  b[i]*x[l[i],t]*(1-rho[h[i],t-1])*x[i,t-1] = 
  b[i]*(1 - rho[h[i],t-1])*x[i,t-1];

# if c=i and j!=h_i, fix R_{j}
subject to fixRj{t in T, i in N, j in M diff {h[i]}} :
  r[j,t]*x[i,t-1] = r[j,t-1]*x[i,t-1];

# if c=0 stop
subject to stop{t in T} : x[0,t]*x[0,t-1] = x[0,t-1];

# if c=0 then fix Rj's
subject to fixRj0{t in T, j in M} : r[j,t]*x[0,t-1] = r[j,t-1]*x[0,t-1];

## definitions
# c takes one value at any time
subject to xass{t in T0} : sum{i in N0} x[i,t] = 1;

# def of rho in terms of r
subject to rhodef1{j in M, t in T0} : r[j,t] - 1 >= rho[j,t];
subject to rhodef2{j in M, t in T0} : r[j,t] - 1 <= (vm+1)*rho[j,t];

