#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>

#ifdef _OPENMP
#include <omp.h>
#endif

#include "gmp.h"
#include "gmp-impl.h"
#include "gmp-mparam.h"

#include "gcd_utils.h"
#include "qseq.h"
#ifndef USE_GMP		
#include "gmp_fft.h"
#endif
#include "gcd_matrix.h"
#include "regular.h"

#include "gcd_euclid.h"
#include "gcd_jebelean.h"
#include "gcd_thull_yap.h"
#ifdef USE_PAN_WANG
#include "gcd_pan_wang.h"
#endif
#if 0 // obsolete?
#include "special_matrix22.h"
#include "waksman.h"
#include "special_matrix22_mul.h"
#endif
#include "git_version.h"

#define TEST_BUILD_PRODUCT 1
#define TEST_QSEQ          2
#define TEST_STRASSEN      3
#define TEST_MATMULVEC     4
#define TEST_MATMUL22      5
#define TEST_GCD           6
#define TEST_HGCD          7
#define TEST_CORNACCHIA    8

#define ALGO_EUCLID      100
#define ALGO_JEBELEAN    101
#define ALGO_THULL_YAP   102
#define ALGO_GMP         103
#define ALGO_QEUCLID     200
#define ALGO_QJEBELEAN   201
#define ALGO_PAN_WANG    202

/* FIXME: Lehmer? */

#define FAMILY_NONE        0
#define FAMILY_MERSENNE    1
#define FAMILY_FIBONACCI   2
#define FAMILY_RANDOM      3
#define FAMILY_RANDOM2     4
/* for Thull Yap */
#define FAMILY_mB1         5
#define FAMILY_mB2         6
#define FAMILY_mC          7
#define FAMILY_mD          8
#define FAMILY_pA          9
#define FAMILY_pB         10

#define DEBUG_TESTS 0
#define DO_PROFILE  0 /* FIXME: what's the use of this, anyway? */

#if ONE_DAY
static void TestStrassen(size_t sizemin, size_t sizemax, size_t sizeincr,
			 size_t sizemul, int ntests)
{
    gmp_randstate_t state;
    double tt, tplain, tstrassen, trandom;
    mpz_t pw0, pw1, w0, w1, v00, v01, v10, v11;
    mp_size_t m;
    mat2x2_t M;

    int i;

    gmp_randinit_default(state);
    gmp_randseed_ui(state, 42);
    mat2x2_init(M);
    mpz_inits(pw0, pw1, w0, w1, v00, v01, v10, v11, NULL);
    for(; sizemin < sizemax;
	sizemin = (sizemul ? sizemul * sizemin : sizemin + sizeincr)){
	m = sizemin;
	for(i = 0; i < ntests; i++){
	    tt = runtime();
	    /* M entries have m bits */
	    mpz_urandomb(M->a, state, m * GMP_NUMB_BITS);
	    mpz_urandomb(M->b, state, m * GMP_NUMB_BITS);
	    mpz_urandomb(M->c, state, m * GMP_NUMB_BITS);
	    mpz_urandomb(M->d, state, m * GMP_NUMB_BITS);
	    /* vectors v0 and v1 have 2*m bits */
	    mpz_urandomb(v00, state, m * GMP_NUMB_BITS);
	    mpz_urandomb(v01, state, m * GMP_NUMB_BITS);
	    mpz_urandomb(v10, state, m * GMP_NUMB_BITS);
	    mpz_urandomb(v11, state, m * GMP_NUMB_BITS);
	    trandom = runtime()-tt;
	    mpz_set_ui(pw0, 0);
	    mpz_set_ui(pw1, 0);
	    tt = runtime();
	    mat2x2_addmul_vec_cut_plain(pw0, pw1, M, v01, v00, v11, v10, m);
	    tplain = runtime()-tt;
	    mpz_set_ui(w0, 0);
	    mpz_set_ui(w1, 0);
	    tt = runtime();
	    mat2x2_addmul_vec_cut_Strassen(w0, w1, M, v01, v00, v11, v10, m);
	    tstrassen = runtime()-tt;
	    printf("Random  : %lu %lf\n", m, trandom);
	    printf("Plain   : %lu %lf\n", m, tplain);
	    printf("Strassen: %lu %lf\n", m, tstrassen);
# if 0
	    gmp_printf("pw0:=%Zd; w0:=%Zd;\n", pw0, w0);
	    gmp_printf("pw1:=%Zd; w1:=%Zd;\n", pw1, w1);
# endif
	    assert(mpz_cmp(pw0, w0) == 0);
	    assert(mpz_cmp(pw1, w1) == 0);
	}
    }
    gmp_randclear(state);
    mpz_clears(pw0, pw1, w0, w1, v00, v01, v10, v11, NULL);
    mat2x2_clear(M);
}
#endif /* ONE_DAY */

static void TestMatMul22(size_t sizemin, size_t sizemax, size_t sizeincr,
			 size_t sizemul, int gmin, int gmax, int ntests)
{
    gmp_randstate_t state;
    mp_ptr n[4], r[4], m[4], p[4], op[4], tp;
    mp_size_t nl, ml, pl, tpn;
    double tt, tplain;
#ifndef USE_GMP		
    double tfft;
#endif
    int i, g, nt;
    
    gmp_randinit_default(state);
    gmp_randseed_ui(state, 42);
    for(; sizemin < sizemax;
	sizemin = (sizemul ? sizemul * sizemin : sizemin + sizeincr)){
	ml = sizemin;
	for(g = gmin; g < gmax; g++){
	    nl = g * ml;
	    pl = nl + ml + 1;
	    for(i = 0; i < 4; i++){
		n[i]  = (mp_ptr)malloc(nl * sizeof(mp_limb_t));
		m[i]  = (mp_ptr)malloc(ml * sizeof(mp_limb_t));
		p[i]  = (mp_ptr)malloc(pl * sizeof(mp_limb_t));
		r[i]  = (mp_ptr)malloc(pl * sizeof(mp_limb_t));
		op[i] = (mp_ptr)malloc(pl * sizeof(mp_limb_t));
	    }
	    tpn = mpn_matrix22_mul_itch(nl, ml);
	    tp = (mp_ptr)malloc(tpn * sizeof(mp_limb_t));
	    tplain = 0.0;
#ifndef USE_GMP		
	    tfft = 0.0;
#endif
	    for(nt = 0; nt < ntests; nt++){
		/* n contains nl words, m contains ml words */
		for(i = 0; i < 4; i++){
		    do {
			mpn_random(n[i], nl);
		    } while(n[i][nl-1] == 0);
		    MPN_COPY(r[i], n[i], nl);
		    MPN_ZERO(r[i]+nl, pl-nl);
		    do {
			mpn_random(m[i], ml);
		    } while(m[i][ml-1] == 0);
		    MPN_ZERO(op[i], pl);
		}
		tt = runtime();
		mpn_matrix22_mul(r[0], r[1], r[2], r[3], nl,
				 m[0], m[1], m[2], m[3], ml, tp);
		tplain += runtime()-tt;
#ifndef USE_GMP		
		tt = runtime();
		mpn_mat_mul_mat_fft(op, pl, n, nl, m, ml);
		tfft += runtime()-tt;
		for(i = 0; i < 4; i++)
		    if(mpn_cmp(r[i], op[i], pl) != 0)
			printf("r[%d] badk\n", i);
#endif
	    }
#ifndef USE_GMP		
	    printf("%lu %lf %lf\n", sizemin, tplain/ntests, tfft/ntests);
#else
	    printf("%lu %lf\n", sizemin, tplain/ntests);
#endif
	    for(i = 0; i < 4; i++){
		free(n[i]);
		free(r[i]);
		free(m[i]);
		free(p[i]);
		free(op[i]);
	    }
	    free(tp);
	}
    }
    gmp_randclear(state);
}

static void TestMatMulVec(size_t sizemin, size_t sizemax, size_t sizeincr,
			 size_t sizemul, int gmin, int gmax, int ntests)
{
    mpz_t zv[2], z;
    gmp_randstate_t state;
    double tt, totp, totw[100], totf;
    mp_ptr v[2], w[2], ww[2], M[4], fw[4], A[4];
    mp_size_t vn[2], wn[2], wwn[2], Mn[4], fwn[4];
    mp_size_t m, nA = 0, nv = 0;
    int i, n3, g, nt; /* for info: n1 = 2, n2 = 2 */

    gmp_randinit_default(state);
    gmp_randseed_ui(state, 42);
    for(i = 0; i < 2; i++)
	mpz_init(zv[i]);
    mpz_init(z);
    for(; sizemin < sizemax;
	sizemin = (sizemul ? sizemul * sizemin : sizemin + sizeincr)){
	m = sizemin;
	for(g = gmin; g < gmax; g++){
	    totp = 0.0;
	    totf = 0.0;
	    for(n3 = 2; n3 <= g; n3++)
		totw[n3] = 0.0;
	    for(i = 0; i < 2; i++){
		/* w[i] = M->? * v[0] + M->? * v[1]
		   hence length is m+g*m+1
		 */
		w[i]  = (mp_ptr)malloc(((g+1)*m+1) * sizeof(mp_limb_t));
		fw[i] = (mp_ptr)malloc(((g+1)*m+1) * sizeof(mp_limb_t));
		ww[i] = (mp_ptr)malloc(((g+1)*m+1) * sizeof(mp_limb_t));
	    }
	    if(g == 2){
		nA = m;
		nv = g*m;
		/* nA+nv+1 = 3*m+1, w[i] => 3*m+1 */
		for(i = 0; i < 4; i++)
		    A[i] = (mp_ptr)malloc((nA+nv+1) * sizeof(mp_limb_t));
	    }
	    for(nt = 0; nt < ntests; nt++){
		/* M entries have m words */
		for(i = 0; i < 4; i++){
		    /* M[i] has exactly m words */
		    do{
			mpz_urandomb(z, state, m * GMP_NUMB_BITS);
		    } while((mp_size_t)mpz_size(z) < m);
		    M[i]  = PTR(z);
		    Mn[i] = m;
		}
		/* vectors v0 and v1 have exactly g*m words */
		for(i = 0; i < 2; i++){
		    do{
			mpz_urandomb(zv[i], state, (g*m) * GMP_NUMB_BITS);
		    } while((mp_size_t)mpz_size(zv[i]) < g*m);
		    v[i] = zv[i]->_mp_d;
		    vn[i] = g*m;
		}
		tt = runtime();
		mat2x2_mul_vec_mpn_plain(w, wn, M, Mn, v, vn);
		totp += runtime()-tt;
#ifndef USE_GMP		
		tt = runtime();
		mul_vec_mpn_fft(fw, fwn, M, Mn, v, vn);
		totf += runtime()-tt;
		for(i = 0; i < 2; i++)
		    assert(fwn[i] == wn[i]
			   && mpn_cmp(fw[i], w[i], wn[i]) == 0);
#endif
		if(g == 2){
		    /* same alloc for A[i] with size nA, v[i] with nv */
		    for(i = 0; i < 4; i++){
			MPN_ZERO(A[i], nA+nv+1);
			MPN_COPY(A[i], M[i], Mn[i]);
		    }
		    for(i = 0; i < 2; i++)
			wwn[i] = nA+nv+1;
		    tt = runtime();
		    mat2x2_mul_vec_mpn222(ww, wwn, A, nA, v, nv);
		    totw[g] += runtime()-tt;
		    for(i = 0; i < 2; i++)
			assert(wwn[i] == wn[i]
			       && mpn_cmp(ww[i], w[i], wn[i]) == 0);
		}
#if 0
		else if(g > 1)
		    for(n3 = g; n3 <= g; n3++){
			tt = runtime();
			mat2x2_mul_vec_cut_mpn(ww, wwn, M, Mn, v, vn, n3);
			totw[n3] += runtime()-tt;
			for(i = 0; i < 2; i++)
			    assert(wwn[i] == wn[i]
				   && mpn_cmp(ww[i], w[i], wn[i]) == 0);
		    }
#endif
	    }
	    printf("m = %lu g = %d plain = %2.2lf fft = %2.2lf",
		   m, g, totp/ntests, totf/ntests);
	    if(g > 1){
		if(g == 2)
		    printf(" 2x2x2 =");
		else
		    printf(" waksman =");
		for(n3 = g; n3 <= g; n3++)
		    printf(" %2.2lf", totw[n3]/ntests);
	    }
	    printf("\n");
	    for(i = 0; i < 2; i++){
		free(w[i]); free(fw[i]); free(ww[i]);
	    }
	    if(g == 2)
		for(i = 0; i < 4; i++)
		    free(A[i]);
	} /* for g */
    } /* for ntest */
    gmp_randclear(state);
    for(i = 0; i < 2; i++)
	mpz_clear(zv[i]);
    mpz_clear(z);
}

#ifdef REGULAR /* TODO: move qseq_build* in qseq.c */
static void TestBuildProduct(size_t nqmin, size_t nqmax, size_t nqincr,
			     size_t nqmul, int ntests)
{
    double tt, tplain, tplain_mpn, ttree, tmin = 0.0;
    /*    double trandom;*/
    qseq_t R;
    struct hgcd_matrix Mplain, Mplain_mpn, Mtree;
    size_t i, len, lenmin, nq;
    int j;

    srandom(42);
#if 0
    printf("nq:rand:plain:plain_mpn:tree (plain/tree plain_mpn/tree) "
	   "[repeated %d times]\n", ntests);
#endif
    for(; nqmin < nqmax;
	nqmin = (nqmul ? nqmul * nqmin : nqmin + nqincr)){
	nq = nqmin;
	tt = runtime();
	qseq_init(R, nq);
	/* adapting to the law for partial quotients */
	for(i = 0; i < nq; i++){
	    mp_limb_t r = (mp_limb_t)(labs(random()) % 1000);
	    if(r <= 415)
		r = 1;
	    else if(r <= 585) /* 415+170 */
		r = 2;
	    else if(r <= 678) /* 415+170+93 */
		r = 3;
	    else if(r <= 737) /* 415+170+93+59 */
		r = 4;
	    else
		r = 5 + (labs(random()) % 100);
	    qseq_add_last_ui(R, r);
	}
	/*	trandom = runtime()-tt;*/
	tplain = 0.0;
	tt = runtime();
	for(j = 0; j < ntests; j++){
	    qseq_build_product_plain(&Mplain, R);
	    if(j < ntests-1)
		hgcd_matrix_clear(&Mplain);
	}
	    
	tplain = runtime()-tt;
	printf("plain: %d %3.3lf\n", qseq_card(R), tplain);

	tplain_mpn = 0.0;
	tt = runtime();
	for(j = 0; j < ntests; j++){
	    qseq_build_product_plain_mpn(&Mplain_mpn, R);
	    if(j < ntests-1)
		hgcd_matrix_clear(&Mplain_mpn);
	}
	tplain_mpn = runtime()-tt;
	printf("plain_mpn: %d %3.3lf\n", qseq_card(R), tplain_mpn);
	if(mat_is_equal(&Mplain_mpn, &Mplain) == 0){
	    printf("#!# error: mpn != plain\n");
	    assert(0);
	}

	lenmin = 0;
	for(len = 10; len <= nq; len += 10){
	    ttree  = 0.0;
	    tt = runtime();
	    for(j = 0; j < ntests; j++){
		qseq_build_product_tree(&Mtree, R, len);
		if(j < ntests-1)
		    hgcd_matrix_clear(&Mtree);
	    }
	    ttree = runtime()-tt;
	    if(lenmin == 0 || ttree < tmin){
		lenmin = len;
		tmin = ttree;
	    }
	    if(mat_is_equal(&Mtree, &Mplain) == 0){
		printf("#!# error: tree != plain\n");
		assert(0);
	    }
	    hgcd_matrix_clear(&Mtree);
#if 1
	    printf("W tree: %d %lu %3.3lf\n", qseq_card(R), len, ttree);
#endif
	}
	printf("tree: %d %lu %3.3lf\n", qseq_card(R), lenmin, tmin);
#if 0
	printf("#T# %lu %lf %lf %lu:%lf %lf (%lf %lf)\n",
	       qseq_card(R), trandom, tplain, len, tplain_mpn, ttree,
	       tplain/ttree, tplain_mpn/ttree);
#endif
	qseq_clear(R);
	hgcd_matrix_clear(&Mplain);
	hgcd_matrix_clear(&Mplain_mpn);
    }
}
#endif /* REGULAR */

static void TestQseq()
{
    qseq_t Q;
    mp_limb_t a[10] = {1, 2, 3, 4, 5, 6, 7, 9, 10};
    int i, ii;

    printf("QSEQ_DATA_TYPE = %d\n", QSEQ_DATA_TYPE);
    qseq_init(Q, 2);
    qseq_dump(Q);
    for(i = 0; i < 4; i++){
	qseq_add_last_mpn(Q, a, i+1);
	printf("Q_%d:=", i); qseq_print(Q);
	qseq_dump(Q);
	qseq_add_last_mpn(Q, a+i, 1);
	printf("Q_%d:=", i); qseq_print(Q);
	qseq_dump(Q);
    }
    ii = i;
    for(i = 0; i < 2; i++, ii++){
	qseq_remove_last(Q);
	printf("Q_%d:=", ii); qseq_print(Q);
	qseq_dump(Q);
    }
    for(i = 5; i < 10; i++, ii++){
	qseq_add_last_mpn(Q, a+i, 1);
	printf("Q_%d:=", ii); qseq_print(Q);
	qseq_dump(Q);
    }
    /* final removals */
    for( ; qseq_is_empty(Q) == 0; ii--){
	qseq_remove_last(Q);
	printf("Q_%d:=", ii); qseq_print(Q);
	qseq_dump(Q);
    }
    qseq_clear(Q);
}

/* INPUT: a > b, rmin > 0; tp[0] = h when needed...
   PRECONDITION: size(a) > an, size(b) > bn.
   OUTPUT: bn
   POSTCONDITION: a >= rmin > b; when rmin = 1, a is the gcd.
*/
static mp_size_t test_gcd_aux(regular_t R,
			      mp_ptr a, mp_size_t an,
			      mp_ptr b, mp_size_t bn,
			      mp_ptr emin, mp_size_t eminn,
			      mp_ptr tp, mp_size_t tp_alloc,
			      int test_type, int algo, double *tot)
{
    mp_ptr ri, rip1;
    mp_size_t gn = 0;
    double tt;

    switch(algo){
    case ALGO_EUCLID:
	tt = runtime();
	euclid_gcd(R, a, an, b, bn, 0, emin, eminn, tp, tp_alloc, 0);
	*tot += runtime()-tt;
	gn = a[an];
	break;
    case ALGO_QEUCLID:
        tt = runtime();
	euclid_gcd(R, a, an, b, bn, 0, emin, eminn, tp, tp_alloc, 1);
        *tot +=runtime()-tt;
	gn = a[an];
	break;
    case ALGO_JEBELEAN:
        tt = runtime();
	jebelean_gcd(R, a, an, b, bn, 0, emin, eminn, tp, tp_alloc, 0);
	*tot +=runtime()-tt;
	gn = a[an];
	break;
    case ALGO_QJEBELEAN:
        tt = runtime();
	jebelean_gcd(R, a, an, b, bn, 0, emin, eminn, tp, tp_alloc, 1);
	gn = a[an];
        *tot +=runtime()-tt;
	break;
    case ALGO_THULL_YAP:
	if(test_type == TEST_GCD){
	    int dolastmul = 1;
	    
	    tt = runtime();
	    gn = thull_yap_gcd(R, tp, a, an, b, bn, dolastmul);
	    *tot +=runtime()-tt;
	    MPN_COPY(a, tp, gn);
	}
	else if(test_type == TEST_HGCD){
	    ri = (mp_ptr)malloc((an+1) * sizeof(mp_limb_t));
	    rip1 = (mp_ptr)malloc((an+1) * sizeof(mp_limb_t));
	    tt = runtime();
	    thull_yap_use(ri, rip1, a, an, b, bn, emin, eminn, tp, tp_alloc);
            *tot +=runtime()-tt;
	    /* ri >= emin > rip1 */
	    assert((mp_size_t)ri[an] > eminn
		   || ((mp_size_t)ri[an] == eminn
		       && mpn_cmp(ri, emin, eminn) >= 0));
	    assert(eminn > (mp_size_t)rip1[an]
		   || ((mp_size_t)rip1[an] == eminn
		       && mpn_cmp(emin, rip1, eminn) > 0));
#if 0
	    printf("ty_ri:="); MPN_PRINT(ri, ri[an]); printf(";\n");
	    printf("ty_rip1:="); MPN_PRINT(rip1, rip1[an]); printf(";\n");
#endif
	    /* a >= rmin > b */
	    MPN_COPY(a, ri, (long)ri[an]);
	    a[an] = ri[an];
	    gn = a[an];
	    MPN_COPY(b, rip1, (long)rip1[an]);
	    free(rip1);
	    free(ri);
	}
	break;
    case ALGO_GMP:
	/* an >= bn > 0; a and b are destroyed; a or b must be odd */
	assert((a[0] & 1) == 1 || (b[0] & 1) == 1);
	tt = runtime();
	gn = mpn_gcd(tp, a, an, b, bn);
	*tot +=runtime()-tt;
	MPN_COPY(a, tp, gn);
	break;
#ifdef USE_PAN_WANG
    case ALGO_PAN_WANG:
	if(test_type == TEST_GCD){
	    mp_size_t h = an+1;
	    
            hgcd_matrix_set_identity(R->Q); /* FIXME: why? */
	    pan_wang(R, a, an, b, bn, h, tp, tp_alloc, 1);
	    gn = a[an];
	}	
        else if(test_type == TEST_HGCD){
	    mp_size_t h = tp[0];

	    tp[0] = 0; /* who cares, really? */
	    hgcd_matrix_set_identity(R->Q);
	    pan_wang(R, a, an, b, bn, h, tp, tp_alloc, 1);
	    /* check that ||Q_i|| <= base^h < ||Q_{i+1}|| */
	}
	break;
#endif
    default:
	break;
    }
    return gn;
}

/* INPUT: A > B; 
          rmin = 1 for gcd test with gcd in G; 
	  if rmin > 1, G is expected remainder.
   POSTCONDITION: ri >= rmin > rip1.
*/
static void TestGCD(mpz_t ri, mpz_t rip1, mpz_t A, mpz_t B,
		    mpz_t G, mpz_t rmin, mp_size_t h,
		    int test_type, int test_algo, double *tot)
{
    regular_t R;
    mp_size_t an = mpz_size(A), bn = mpz_size(B), n = an, tp_alloc = 2*n+3;
    mp_ptr a = (mp_ptr)malloc((n+1) * sizeof(mp_limb_t));
    mp_ptr b = (mp_ptr)malloc((n+1) * sizeof(mp_limb_t));
    mp_limb_t *tp = (mp_ptr)malloc(tp_alloc * sizeof(mp_limb_t));
    int uselq = 1;

    MPN_SET_MPZ(a, an, A);
    a[an] = 0;
    MPN_SET_MPZ(b, bn, B);
    /* clear b[bn..an+1[ */
    MPN_ZERO(b+bn, an-bn+1);

    tp[0] = h; /* when needed */

    mpz_set_ui(ri, 0);
    mpz_set_ui(rip1, 0);
    /* TODO: only in theses cases? Odd */
    regular_init(R, an, an << 1, uselq, 1); /* FIXME */
    a[n] = test_gcd_aux(R, a, an, b, bn, PTR(rmin), SIZ(rmin),
			tp, tp_alloc, test_type, test_algo, tot);
    if(test_algo != ALGO_PAN_WANG){
	MPZ_SET_MPN(ri, a, a[n]);
	MPZ_SET_MPN(rip1, b, b[n]);
	/* ri >= rmin > rip1 */
	/* if rmin = 1, then rip1 = 0 and gcd is in ri */
	if(G != NULL){
	    if(mpz_cmp(ri, G) != 0){
		if(test_algo != ALGO_QEUCLID && test_algo != ALGO_QJEBELEAN
		   && test_algo != ALGO_PAN_WANG){
		    printf("G != ri: %d\n", mpz_cmp(G, ri));
		    printf("Error: we should have G=ri >= rmin > r\n");
		    gmp_printf("G:=%Zd;\nri:=%Zd;\nrmin:=%Zd;\nr:=%Zd;\n",
			       G, ri, rmin, rip1);
		}
		else{
		    printf("Error: we should have G=xi!\n");
		    gmp_printf("G:=%Zd;\nxi:=%Zd;\nxip1:=%Zd;\n", G, ri, rip1);
		}
		assert(0);
	    }
	    else
		fprintf(stderr, "Expected result\n");
	}
    }
    if(test_algo != ALGO_GMP && (G == NULL || test_algo == ALGO_PAN_WANG)){
	/* if G == NULL, we use plain Euclid to compare */
	mp_limb_t emin[1];
	mp_size_t eminn = 0;
	regular_t S;
	int ok = 1, uselq = 1, qgcd = 0;

	if(test_algo == ALGO_PAN_WANG){
	    uselq = 1; /* really? */
	    qgcd = 1;
	}
	else{
	    emin[0] = 1;
	    eminn = 1;
	}
	MPN_SET_MPZ(a, an, A);
	a[an] = 0;
	MPN_SET_MPZ(b, bn, B);
	/* clear b[bn..an+1[ */
	MPN_ZERO(b+bn, an-bn+1);
	regular_init(S, an, an << 1, 1, uselq); /* FIXME */
	euclid_gcd(S, a, an, b, bn, h, emin, eminn, tp, tp_alloc, qgcd);
	/* to be sure */
	if(qseq_is_used(R->lq) != 0){
	    /* check lq's */
	    if(qseq_is_equal(R->lq, S->lq) != 0)
		printf("lq's coincide\n");
	    else{
		printf("lq's do not coincide\n");
		printf("R->lq:="); qseq_print(R->lq); printf(";\n");
		printf("S->lq:="); qseq_print(S->lq); printf(";\n");
		ok = 0;
	    }
	}
	if(mat_is_equal(R->Q, S->Q))
	    printf("matrices coincide\n");
	else{
	    printf("Incorrect matrices\n");
	    printf("R->Q:="); hgcd_matrix_print(R->Q); printf(";\n");
	    printf("S->Q:="); hgcd_matrix_print(S->Q); printf(";\n");
	    ok = 0;
	}
	assert(ok);
	regular_clear(S);
    }
    free(a);
    free(b);
    free(tp);
    regular_clear(R);
}

static void qseq_add_last_mpz(qseq_t M, mpz_t x)
{
    assert(mpz_sgn(x) != 0);
    qseq_add_last_mpn(M, x->_mp_d, mpz_size(x));   
}

/** Develop u/v in continued fractions: stop when ri >= rmin > r.
    SIDE-EFFECT: M is updated s.t. (u, v) = M (ri, r).
    REM: u and v are *not* modified.
*/
static void mpz_gcd_euclid(mpz_t ri, mpz_t r, qseq_t lq, mpz_t u, mpz_t v, mpz_t rmin)
{
    mpz_t A, B, q, R;

    mpz_inits(q, R, A, B, NULL);
    mpz_set(A, u);
    mpz_set(B, v);
    while(mpz_cmp(B, rmin) >= 0){
#if 1 /* do we have q = 1? */
	mpz_sub(R, A, B);
	if(mpz_cmp(R, B) < 0){
	    /* R = A - B < B */
	    if(lq != NULL){
		mp_limb_t qt = 1;
		qseq_add_last_mpn(lq, &qt, 1);
	    }
	}
	else{
# if 0
	    int done = 0;
	    mp_limb_t qq;

	    for(qq = 2; qq <= 2; qq++){
		mpz_sub(R, R, B);
		if(mpz_cmp(R, B) < 0){
		    /* R = A - qq * B < B */
		    if(lq != NULL)
			qseq_add_last_ui(lq, qq);
		    done = 1;
		    break;
		}
	    }
	    if(done == 0){
# else
	    /* do we have q = 2? */
	    mpz_sub(R, R, B);
	    if(mpz_cmp(R, B) < 0){
		/* R = A - 2*B < B */
		if(lq != NULL){
		    mp_limb_t q = 2;
		    qseq_add_last_ui(lq, q);
		}
	    }
	    else{
# endif
#endif
		/* ordinary euclidean step */
		mpz_tdiv_qr(q, R, A, B);
		if(lq != NULL)
		    qseq_add_last_mpz(lq, q);
#if 1
	    }
	}
#endif
	mpz_swap(A, B); mpz_swap(B, R);
    }
    /* A >= rmin > B */
    mpz_set(ri, A);
    mpz_set(r, B);
    mpz_clears(A, B, q, R, NULL);
}

/** INPUT: A > B > 0 and same size. */
static void TestHGCD(mpz_t A, mpz_t B, MAYBE_UNUSED char *gamma, int algo,
		     int verbose, double *tot)
{
    mpz_t rac, AA, BB, eucprev, eucr, r, ri;
    mp_size_t h = 11;
    qseq_t lq;

    mpz_inits(rac, AA, BB, eucprev, eucr, ri, r, NULL);
    if(algo != ALGO_PAN_WANG){
	/* simulate Cornacchia */
	mpz_sqrt(rac, A);
	if(verbose)
	    gmp_printf("A:=%Zd;\nB:=%Zd;\nrac:=%Zd;\n", A, B, rac);
	/* reference */
	mpz_set(AA, A); mpz_set(BB, B);
	qseq_init(lq, mpz_size(A));
	mpz_gcd_euclid(eucprev, eucr, lq, AA, BB, rac);
	if(verbose > 2){
	    gmp_printf("Euclid_ri:=%Zd;\n", eucprev);
	    gmp_printf("Euclid_rip1:=%Zd;\n", eucr);
	    printf("zlq:="); qseq_print(lq); printf(";\n");
	}
	qseq_clear(lq);
    }
    else{
	/* TODO: reference */
#if 0
	mpz_set(AA, A); mpz_set(BB, B);
	mpz_gcd_euclid(eucprev, eucr, NULL, AA, BB, rac);
	if(verbose){
	    gmp_printf("Euclid_ri:=%Zd;\n", eucprev);
	    gmp_printf("Euclid_rip1:=%Zd;\n", eucr);
	}
#endif
    }
    mpz_set(AA, A); mpz_set(BB, B);
    TestGCD(ri, r, AA, BB, eucprev, rac, h, TEST_HGCD, algo, tot);
    mpz_clears(rac, AA, BB, eucprev, eucr, ri, r, NULL);
}

static void TestCornacchia(int algo, int family, mp_limb_t D, int yone,
			   size_t sizemin, size_t sizemax, size_t sizeincr,
			   size_t sizemul, int ntests, int verbose)
{
    gmp_randstate_t state;
    mpz_t p, rac, x, y, sqrtd, ri, r, N;
    mp_limb_t d = ((D & 3) == 0 ? D/4 : D);
    mp_bitcnt_t nb;
    double tot;
    int i;

    gmp_randinit_default(state);
    gmp_randseed_ui(state, 42);
    mpz_inits(p, x, y, rac, sqrtd, ri, r, N, NULL);
    for(; sizemin < sizemax;
	sizemin = (sizemul ? sizemul * sizemin : sizemin + sizeincr)){
	nb = (sizemin / 2) * GMP_NUMB_BITS;
	tot = 0.0;
	mpz_set_ui(N, 1);
	mpz_mul_2exp(N, N, nb);
	for(i = 0; i < ntests; i++){
	    do {
		switch(family){
		case FAMILY_RANDOM:
		    mpz_urandomb(x, state, nb);
		    if(yone)
			mpz_set_ui(y, 1);
		    else
			mpz_urandomb(y, state, nb);
		    break;
		case FAMILY_RANDOM2:
		    mpz_urandomm(x, state, N);
		    if(yone)
			mpz_set_ui(y, 1);
		    else
			mpz_urandomm(y, state, N);
		    break;
		}
		if((d & 1) == 0){
		    /* x^2+2*d'*y^2 => x odd */
		    if(mpz_even_p(x))
			mpz_add_ui(x, x, 1);
		}
		else{
		    /* x^2+(2*d'+1)*y^2 */
		    if(mpz_odd_p(x) && mpz_odd_p(y))
			mpz_add_ui(x, x, 1);
		    if(mpz_even_p(x) && mpz_even_p(y))
			mpz_add_ui(x, x, 1);
		}
		mpz_mul(p, x, x);
		mpz_mul(sqrtd, y, y);
		mpz_mul_ui(sqrtd, sqrtd, d);
		mpz_add(p, p, sqrtd);
		if(verbose >= 2){
		    gmp_printf("x:=%Zd;\n", x);
		    gmp_printf("y:=%Zd;\n", y);
		    gmp_printf("p:=%Zd;\n", p);
		    printf("p-(x^2+%lu*y^2);\n", d);
		    printf("// size(p)=%lu\n", mpz_size(p));
		}
	    } while(mpz_probab_prime_p(p, 1) == 0);
	    /* sqrt(-d) = x/y mod p */
	    mpz_invert(sqrtd, y, p);
	    mpz_mul(sqrtd, sqrtd, x);
	    mpz_mod(sqrtd, sqrtd, p);
	    mpz_add(rac, sqrtd, sqrtd);
	    if(mpz_cmp(rac, p) < 0)
		mpz_sub(sqrtd, p, sqrtd);
	    /* call HGCD */
	    mpz_sqrt(rac, p);
	    TestGCD(ri, r, p, sqrtd, NULL, rac, 0, TEST_HGCD, algo, &tot);
	    if(verbose >= 2)
		gmp_printf("ri:=%Zd;\nr:=%Zd;\n", ri, r);
	    /* ri >= rac = floor(sqrt(p)) > r
	       if ri = rac, then we must take r = ri
	     */
	    if(mpz_cmp(ri, rac) == 0){
		printf("W: ri == rac\n");
		mpz_set(r, ri);
	    }
	    /* are we sure to get x or y or anyone of these? */
	    if(mpz_cmp(x, r) == 0){
		if(verbose >= 1)
		    printf("r = x\n");
	    }
	    else if(d == 1 && mpz_cmp(y, r) == 0){
		if(verbose >= 1)
		    printf("r = y\n");
	    }
	    else
		assert(0);
	}
	printf("%lu %lf\n", sizemin, tot/ntests);
    }
    mpz_clears(p, x, y, rac, sqrtd, ri, r, N, NULL);
}

#undef _OPENMP

/* INPUT: nmin >= 1. */
static void UseFibonacci(int test_type, int algo, int nmin, int nmax, int verbose){
    mpz_t Fn, Fnm1, Fnp1, G, rmin, ri, r;
    double tot = 0.0;
    int n;

    mpz_inits(Fn, Fnm1, Fnp1, G, rmin, ri, r, NULL);
    mpz_set_ui(rmin, 0);
    mpz_set_ui(G, 1);
    mpz_fib2_ui(Fn, Fnm1, nmin);
    for(n = nmin+1; n < nmax; n++){
	/* F[n] = F[n-1]+F[n-2] */
	mpz_add(Fnp1, Fn, Fnm1);
	mpz_swap(Fnm1, Fn);
	mpz_swap(Fn, Fnp1);
	if(test_type == TEST_GCD){
	    printf("Gcd(F_%d, F_%d)\n", n, n-1);
	    TestGCD(ri, r, Fn, Fnm1, G, rmin, 0, test_type, algo, &tot);
	}
	else if(test_type == TEST_HGCD){
	    printf("HGCD(F_%d, F_%d)\n", n, n-1);
	    TestHGCD(Fn, Fnm1, "", algo, verbose, &tot);
	}
    }
    mpz_clears(Fn, Fnm1, Fnp1, G, rmin, ri, r, NULL);
}

static void ComputeMersenne(mpz_t N, int n)
{
    mpz_set_ui(N, 1);
    mpz_mul_2exp(N, N, n);
    mpz_sub_ui(N, N, 1);
}

static int SmallGcd(int a, int b)
{
    int r;

    while(b){
	r = a % b;
	a = b;
	b = r;
    }
    return a;
}

/* INPUT: nmin >= 1. */
static void UseMersenne(int test_type, int algo, int nmin, int nmax, int verbose)
{
    mpz_t Mn, Mk, G, rmin, ri, r;
    double tot = 0.0;
    int n, k;

    mpz_inits(Mn, Mk, G, rmin, ri, r, NULL);
    mpz_set_ui(rmin, 0);
    for(n = nmin; n < nmax; n++){
	ComputeMersenne(Mn, n);
	for(k = nmin; k < n; k++){
	    ComputeMersenne(Mk, k);
	    if(test_type == TEST_GCD){
		ComputeMersenne(G, SmallGcd(n, k));
                printf("GCD(M_%d, M_%d)\n", n, k);
		TestGCD(ri, r, Mn, Mk, G, rmin, 0, test_type, algo, &tot);
	    }
	    else if(test_type == TEST_HGCD){
		printf("HGCD(M_%d, M_%d)\n", n, k);
		TestHGCD(Mn, Mk, "", algo, verbose, &tot);
	    }
	}
    }
    mpz_clears(Mn, Mk, G, rmin, ri, r, NULL);
}

static void GenerateRandom(mp_ptr a, mp_size_t an, mp_ptr b, mp_size_t bn,
			   int family)
{
    switch(family){
    case FAMILY_RANDOM:
	do {
	    mpn_random(a, an);
	} while(a[an-1] == 0);
	do {
	    mpn_random(b, bn);
	} while(b[bn-1] == 0);
	break;
    case FAMILY_RANDOM2:
	do {
	    mpn_random2(a, an);
	} while(a[an-1] == 0);
	do { 
	    mpn_random2(b, bn);
	} while(b[bn-1] == 0);
	break;
    default:
	fprintf(stderr, "Unknown random family: %d\n", family);
	break;
    }
    a[an] = 0;
    b[bn] = 0;
    //	    trandom = runtime()-tt;
    /* make a odd if needed */
    if((a[0] & 1) == 0 && (b[0] & 1) == 0)
	a[0] |= 1;
}

static void UseRandom(int test_type, int family, int algo,
		      size_t sizemin, size_t sizemax, size_t sizeincr,
		      size_t sizemul, int ntests, int verbose)
{
    gmp_randstate_t state;
    mpz_t A, B, ri, rip1, rmin;
    mp_ptr a, b, g, aa, bb, gg, tp;
    mp_size_t an, bn, tp_alloc = 2*sizemax + 3;
    double tot;
    mp_size_t m, h = 0;
    int i;

    gmp_randinit_default(state);
    gmp_randseed_ui(state, 42);
    a = (mp_ptr)malloc((sizemax + 1) * sizeof(mp_limb_t));
    b = (mp_ptr)malloc((sizemax + 1) * sizeof(mp_limb_t));
    g = (mp_ptr)malloc((sizemax + 1) * sizeof(mp_limb_t));
    aa = (mp_ptr)malloc((sizemax + 1) * sizeof(mp_limb_t));
    bb = (mp_ptr)malloc((sizemax + 1) * sizeof(mp_limb_t));
    gg = (mp_ptr)malloc((sizemax + 1) * sizeof(mp_limb_t));
    tp = (mp_ptr)malloc(tp_alloc * sizeof(mp_limb_t));
    mpz_init_set_ui(A, 0);
    mpz_init_set_ui(B, 0);
    mpz_init_set_ui(rmin, 1);
    mpz_inits(ri, rip1, NULL);
    for(; sizemin < sizemax;
	sizemin = (sizemul ? sizemul * sizemin : sizemin + sizeincr)){
	tot = 0.0;
	m = sizemin;
	an = m;
	bn = m;
	//	for(i = 0; i < ntests; i++){
	for(i = 0; i < ntests; i++){
	    if(verbose > 0)
		printf("########## test: %lu %d\n", m, i);
	    GenerateRandom(a, an, b, bn, family);
	    /* an >= bn > 0; a and b are destroyed; a or b must be odd */
	    if(mpn_cmp(a, b, an) < 0){
		mp_ptr tmp = a; a = b; b = tmp;
	    }
	    if(verbose){
		printf("a:="); MPN_PRINT(a, an); printf(";\n");
		printf("b:="); MPN_PRINT(b, bn); printf(";\n");
	    }
	    if(test_type == TEST_GCD){
		MPZ_SET_MPN(A, a, an);
		MPZ_SET_MPN(B, b, bn);
		TestGCD(ri, rip1, A, B, NULL, rmin, h, test_type, algo, &tot);
	    }
	    else if(test_type == TEST_HGCD){
		MPZ_SET_MPN(A, a, an);
		MPZ_SET_MPN(B, b, bn);
		TestHGCD(A, B, "", algo, verbose, &tot);
	    }
	}
	printf("%lu %lf\n", m, tot/ntests);
    } /* for */
    mpz_clears(A, B, ri, rip1, rmin, NULL);
    gmp_randclear(state);
    free(a); free(b); free(g);
    free(aa); free(bb); free(gg);
    free(tp);
}

#define RECETTE_GCD_NB 100

static void RecetteGcd(int verbose, int algo, int imin, int imax)
{
    char *data[RECETTE_GCD_NB][5] = {
	/* a = b*q+r s.t. a > b > BASE > r and msb(a_1) = 63
	   r:=BASE-13; b:=11*BASE+20; a:=q*b+r s.t. msb(a_1) = 63
	   q:=398682869717442642; a:=q*b+r;
	   a:=a*BASE^3; b:=b*BASE^2;
	 */
	{"forcing 1st backup = 0 at etiq1", // 0
	 "507807546704825676010135862292204596095976339567133018123876854165679266215562509709816585256960",
	 "69048119089253488408999330993703099846393402037739743870976",
	 "1361129467683753853853498429727072845824", "2/3"},
	{"very unbalanced for etiq4", // 1
	 "507807546704825676010135862292204596095976339567133018123876854165679266215562509709816585256960507807546704825676010135862292204596095976339567133018123876854165679266215562509709816585256960",
	 "69048119089253488408999330993703099846393402037739743870976",
	 "1361129467683753853853498429727072845824", "2/3"},
	{"big 1st quotient", // 2
	 "3138550867742757614355465501059594761568381411785544040448",
	 "3743106036",
	 "148", "2/3"},
	{"gcd with 1", // 3
	 "3138550867742757614355465501059594761568381411785544040448",
	 "1",
	 "1", "2/3"},
	{"gcd with 2", // 4
	 "3138550867742757614355465501059594761568381411785544040448",
	 "2",
	 "2", "2/3"},
	{"gcd(A, A) => etiq3", // 5
	 "507807546704825676010135862292204596095976339567133018123876854165679266215562509709816585256960",
	 "507807546704825676010135862292204596095976339567133018123876854165679266215562509709816585256960",
	 "507807546704825676010135862292204596095976339567133018123876854165679266215562509709816585256960",
	 "2/3"},
	{"bh | ah", /* ah = 2^128-1, bh = 2^64+1 */ // 6
	 "115792089237316195423570985008687907852952535068243408814012129478038859681064",
	 "6277101735386680764177666584017038507548129987701376917265",
	 "1", "2/3"},
	/* ah = 2^128-1, bh = 11, but we start from a+b, a */
	{"etiq5: ahn=2, bhn=1, qpn=2", // 7
	 "115792089237316195423570985008687907856742316257639139181827218835111150639595",
	 "115792089237316195423570985008687907852970195110734505007801097066002137689224",
	 "3", "2/3"},
	{"bug found with random2", // 8
	 "174854949325598555139529442455185716666718954043489254495187227566972111694633052289617099896979630338282778171707830469127745752350617361719442473119604347066711562260197540489552326888776812394257350450526379280372432844218993120162409927344184073111159801976876078418627938882617871481885962552166710003202721510695562086395194871069600737271984197898772378675549743501584854305501426435012164700289697963410581064779072981495132026536241995982984061454060805329955488554909662347874661851940275328739419896890316085252644873878688671937944768093226051084801063647334294060249920897693732948065845155549958170232696299730173699927075150228962442909224078172460432636678675159162946816886908250889175623591155476205134894660807221792215928762892203360172414860436073645185904312309700678613897909216731587816397810229222699220007816421296715444010719034480770217702682200910897031831004065665297602051465431314960389294348670946829246160729704467388348283503397134618969734790861316808669888163679611783757678993359981591482004298622799094255085001389175360801827504141102659267714183802812032731653834044961272787085191549933943923379634107704938088491816861055001809863234778437371395337729295569977833032027343207539264782408650585815714367283681516623363712590318954148134330778869928204810880315175515241630523560559777643252881520804589902639676548542520226990083680728544747466427375806276195735002328836296837663872080182974791453070548114122336050913004015941852033093767333551200442290260935526888029161098413259706335232",
	 "174854949325598555139529442455185716666716946807110879900924809339759425579134781870101956379583001645665625991588620926002359599715828577116009023452696885226914507281215863430807532848317526318857931427876576409201824357445774736420757821865506064516007947459512090464068676537526137363190979835531286364280272130212337952845049442260706781565350601610143895466787068253725046823782590491220357710312375328985197608142554462159868841469798399015919980721759246616734701912482926669240459893581261590939046038793078449378775084529427273801472117427939884732183244615994554392224056594426988092193579336408684690159190428265538804247811547676732017715627949691910696404240990190498912673134318763237659051986279326516237374004544123951606879418807533298860298574268845213443589694150505324521307688071128738262540177178515619524313814738048852141633204832080460694896688263806230681423289232631836587691836861330600315145689652882858277207741105659262106962170876973685868952197991318633745986282734290234467405981390614583402849618534850647021657584565746989573194630366688419012973139259235631610805018689874345525831195874442846487606785389257548441978947345056087732175669026024057949894479001269839097357752810275185655393611852157292805497566525225982931828003380587798249113745664762339480448514210617444827360743836127603638975113520262765129284457965072120764287570056969738954777851163302991322093674090922388751997600239155550318372880974414351417422118321434897364327748104675981176998730872081140346905686930261383577601",
	 "3", "2/3"},
	{"bug found with random2: bad overflow in mpn_hgcd_extend (bs+w[1])", // 9
	 "5312245120820618300056768953256815045210670254128806048451226447318127959038956134145684223762569047175642015996865347130615699825173247948459440316517295564500061658185566444040660162016869696544432617365997702823083013170749454289054916926238552615088005136598893129907532738582952180841545840642831575744844372620321059466640478046264993921202923728002034890844102792837319373639754687035523813940259458041882533534810565732281740869355407218056880859023298108200245073218752296277991376429576247446060930632131086379049002687582624795175716658751530851695355382220466073519320947922455574560294609285136974034704397978045617201428489074625158565063339825489370586350972836329292512843373056292597308853787527067676523415838035805326169782729256333461511629779522082705635163913099276849666205310152161951328230241428570113",
	 "83003830012822160938387014894637735081416722720762594507050413237769412377895123088212374268944584755206639609960955342702984142474358528559474910683324828640033461984923922339467172863217553226900214249272303978346684360930536220348282787467042532025271862023945709025133007841639815563143216659963624817878523566696829479910567784291979053002566105558562520512857768227947494116287950936054665032761218919435338516926054259717820437618292094283056858785695087609519403564757702292871363506357599750266433103210131053235128954953727564645652525011831656641909959539151557964828033601383377725251774241039287162831573966414577267733160479479715691394723552419137299143777294909818939582042884918305038767765678571236489931831941500090258485900126898148930653336724424529450013115530873069765400386733083579603978863016673280",
	 "1", "2/3"},
	{"bug found with random2: bad update of M->n in mat2_2_set_mpz",
	 "98655808808884311862403418289461303083897553065782318683370952893474999423503244950590929261197148743697975357195905232039324580541474564876948556600851170473421889968636973571987244615262195245789325900278870935191047274611100337373802530685930596970104658839152073229635529125653434392612242943825895708232527931237317359196238380911941422148429379018465074408321992091513075592553315070967821627398063534359844581916106390897427727646920152963365097884862763123511218867397672037053610283616993456407991702827328652472512829422889796465424112258200071551940103144548110045965600226418921727204712032304056486867471919365652910764024549718878541515138870013898444514190532867663444256827026433062577470294794304513694150259463053846023827714151402015663622283748632052348076151320742336488742541323131062097145404281860273695056724201963686465888182044247909401094319033940589992199834524843823775267081644411477032582371998736429251255913824738262161750227965668523122520259730478280416729299965410000926320827858982973843014921573311525850461611776744489946144525376870123274433986358177561935480974281869401492062623622360092031036919258018461472272216940854416189503181877968781283952658004952667850780877756843788607941246528467220232934201889543773209857657779354421549159958581537278941486200209359218348464164016788693526596724190701370496694508715271032545147569702571623215433202286197653253684388748039811196906780379118484315841098277780535876624038894077122602068244566019030653011041491253025868919891349660925383554612848920814803779986504335554899478446677914433799831159992287432896761612093590859321769109884477578032034541247205057766830158734796790693846674568807769437843336991469578776308169343169715629874672525438775781578720334733468817923968889695263162696216296388579695520812881270096680479955576265513836487506735325116868001897796409415161415060307879924520315781120",
	 "49327904404442155931201709144730651541948776532891159341685476446737499711751622475295464630598574371848987678597952616019662290270737282438474278300425585236710944984318486785993622307631097622894662950139435467595523637305551209819226576552783174219537562676407020208178443176817997009463946343717799823715827807125299753740136440537321536638308703663139631931786442066067996128427242698258356130595777555311239716971097435441015054484941065371809018601776068971226639312662563731682130668212167717843278033104272616813873645511302199075760592877324209665821592116275103496383120161396648104350615222274124343673835951796465968640000290541441514562136470286674571072386892588375147769891607686819213170875790702436207961093011043389456783699795913641649090083338345809290081002534730684599521770338031114358708117922728569811098894795933404400336871575748416509833705712106478772090033764896087734428285742055532738101954882120056660396175243081981593059750228174205794243749910980901313037112873477376586900475725346015419625343714183648784597008136050735102205422318734157932195894060612516243817298474551931006616585426407252617774927296886200228606965826868892771140135181578255555369471486648115793990244460218338330921818966030646390719573914482457740745165973851842107832258612751931330622428266033666435338475396194522160896650638247429847685285021636845631729257471053802089962661831084348588349557354589722575717753057822323664300825200118695788789568756674742499068348974767720099427974729850309525235170548811846791392447715204152129820089286483571837148856263894172568511673611217122260293355398672046207490245238897374017692448803062459522354409746446313916703349300269809624824236979087050380618325179194022529846987467667555822843479772836959387208371159839117696462179328697394210005947094338734939217697045515697755390485682248309264505367941586606538895220626658347054408925183",
	 "1", "2/3"},
	{"bug found with random2: bsn can be < wn[1] in mpn_hgcd_extend and we need clear bs[bsn...",
	 "50511774110148767673550550164204187178955547169680547165885927881459199704833661414702555781732940156773363382884303478804134185237234977216997660979635799282392007663942130468857469243014243965844134860942781765173803789544846740830718452893413485458649855754089114193200938308316134902182268240196098436216130528113061702362051469511615698829784866904732349080762663180315813353516159279544705565210246939962808612311565472479712693258666187058052053841432925889257555702538275305408558147624611599427338420365230426525378601522652074544676868624314797494242313461421539125387311873241569662590215147042839289071442912069727743148873907090033793039294537832832657914546736537288548929165253226749067507968219127457525295580638207576522639341917785249871347216429409537904575863720019720132047172670357004061698823688404619687256532051965417897735010691229354425053458825317798737543306293581998910070491186095894408082802838177398125880075946987186956954607070338017111948734470724546518255326951539581329267225987259204228562734896858948617460874513318830883511249344322878680887418154142650569552653668714766210318820062881803750058152207836508399726390292581037082541687883087207263081466506933659958112579898908606362344702171693999436241501525062312650951106621700129446654218047857458250546586351750097172617290331413818684471602335325038985678571513719879920825856951760342869608254972750774782089693911135773739253427290431537997682068608180359455866617382292927012821236097606782958492653396496772184483721009421699853719177719546038501625075487995034849813686582244483689449302920972122901885520212437750222560773087585613595930479591987541336694440278012276053318664392681529189000531720546591632326241911809365558193083413033434012878276231822003012750609658897937391142849016412610295578334571293389200984550696807021691419891768295112719896152564980418572239665381634820148230048710657",
	 "3082994025277634745700106821545665721371798533305697458855342277921093731984476404705966539412410898240561729912372038501228893141921080152404642393776599077294434061519905424124601394226943601430916434383714716724720227331596950613701661034548948388721097667275438763758128508403297199458260277707301202460980093818414167080563342761482395862435185093942443540722363151770022221783243959592531336062998494209914752408019060720805124534382646051093613814848646062038662423487504326044361203708430489305864234333801401547140023376295718383390360728662900230671437151716615826286842267917560749586018165739492101920429719261285640125596833063891562865262157026023955919873792846823095854484520920509345944712871675691790827690907778485058829248588945681685288179787963931181062068092463984296225973082494056307958089189726701678735576365394146232076917088075949053636690459581128773097212746967276496496010810878000638239143750075543163240049874489986642327436441234458040254480825038220479904594615300602390556385799245276805580024937804723029319565942013515817048714543455250235208789745701165279569026248145395218985062991831707830217974393158466067785199581037714968820628241051867119832966361530047910339065726550260741036716100932205969655083257714244071120221654679340461084001560321676025443801248355439305974923873624147980728110581452806109011739005060060604228087667499288851218705078807364237925455813890575257569981450090997117697469299234094394984840574025401463942099019413361096233909056117427663439764954916401592565651111571414769257187704568268701243082044838400201357613851006471104244828842270232637747398962711875413488415772647088571125272932490717217468263604683325933469555629785507020775366368002753612709901526248456328209643292122899677436613886360765877886748185299249994921843183573130403496311896614949399409796011301191279978006072465219426247824679452583086325235712",
	 "1", "2/3"},
	{"bug in pang_wang_small",
	 "36695977855841144185773134324833391052744618577525823751054629758080031335686481348738694653316993740963840",
	 "421249166674228746791672110734681729275580381602196445017243910143",
	 "3", "h: 2"}, /* TODO: use this info in test */
	{"XXX", "", "", ""}
    };
    mpz_t A, B, G, rmin, ri, r;
    mp_size_t h;
    double tot = 0.0;
#if 0 /* obsolete? */
    char *gamma;
#endif
    int i;

    if(imax == -1)
	imax = imin+1;
    imax = (imax > RECETTE_GCD_NB ? RECETTE_GCD_NB : imax);
    mpz_inits(A, B, G, rmin, ri, r, NULL);
    for(i = imin; i < imax; i++){
	if(strcmp(data[i][0], "XXX") == 0)
	    break;
	if(verbose){
	    printf("////\n");
	    printf("//////// gcd_%d test %d: %s\n", algo, i, data[i][0]);
	    printf("////\n");
	}
	mpz_set_str(A, data[i][1], 10);
	mpz_set_str(B, data[i][2], 10);
	mpz_set_str(G, data[i][3], 10);
	if(mpz_cmp(A, B) < 0){
	    printf("humf: A < B\n");
	    continue;
	}
#if 0 /* obsolete? */
	/* TODO: get some info on h for instance */
	gamma = data[i][4];
#endif
	if(algo != ALGO_QEUCLID && algo != ALGO_QJEBELEAN
	   && algo != ALGO_PAN_WANG){
	    mpz_set_ui(rmin, 1);
	    h = 0;
	}
	else{
	    /* trick to simulate +infty */
	    mpz_add_ui(rmin, A, 1);
	    /* we want ||Q_k|| <= rmin < ||Q_oo|| */
	    h = 0;
	}
	TestGCD(ri, r, A, B, G, rmin, h, TEST_GCD, algo, &tot);
    }
    mpz_clears(A, B, G, rmin, ri, r, NULL);
}

static void ThullYapTest(char *test, int test_type, int test_algo,
			 int u, int verbose, double *tot)
{
    mpz_t A, B, A0, B0, A1, B1, q1, q2, q3, r2, r3, r4;
    char *gamma = "1/2";
    int n, t;

    mpz_inits(A, B, A0, A1, B0, B1, q1, q2, q3, r2, r3, r4, NULL);
    /* t = 1 + ceil(u/2) */
    t = 1 + (u/2) + (u & 1);
    n = u+2;
    if(test[0] == '-'){
	if(test[1] == 'B'){
	    /* r3:=BASE^t; # norm is t */
	    mpz_set_ui(r3, 1); mpz_mul_2exp(r3, r3, t * GMP_NUMB_BITS);
	    if(test[2] == '1'){
		mpz_set_ui(q3, 2);
		/* q2:=BASE^(u-t-1); */
		mpz_set_ui(q2, 1); mpz_mul_2exp(q2, q2, (u-t-1)*GMP_NUMB_BITS);
		/* r4:=q2+1; */
		mpz_add_ui(r4, q2, 1);
		mpz_set_ui(q1, 1);
		mpz_set_ui(A1, 1); mpz_mul_2exp(A1, A1, n * GMP_NUMB_BITS);
		mpz_set_ui(B1, 0);
	    }
	    else if(test[2] == '2'){
		mpz_set_ui(q3, 1);
		mpz_set_ui(r4, 1);
		/*	q2:=BASE^(u-t-1);*/
		mpz_set_ui(q2, 1); mpz_mul_2exp(q2, q2, (u-t-1)*GMP_NUMB_BITS);
		mpz_set_ui(q1, 1);
		mpz_set_ui(A1, 1); mpz_mul_2exp(A1, A1, n * GMP_NUMB_BITS);
		mpz_set_ui(B1, 0);
	    }
	}
	else if(test[1] == 'C' || test[1] == 'D'){
	    mpz_set_ui(r3, 1); mpz_mul_2exp(r3, r3, t * GMP_NUMB_BITS);
	    mpz_set_ui(q2, 1); mpz_mul_2exp(q2, q2, (u-t-1) * GMP_NUMB_BITS);
	    mpz_set_ui(q1, 1);
	    mpz_set_ui(A1, 1); mpz_mul_2exp(A1, A1, n * GMP_NUMB_BITS);
	    mpz_set_ui(B1, 1);
	    if(test[1] == 'C'){
		mpz_set_ui(q3, 2);
		mpz_set_ui(r4, 0);
	    }
	    else if(test[1] == 'D'){
		mpz_set_ui(q3, 1);
		mpz_set_ui(r4, 1);
	    }
	}
	mpz_mul(r2, q3, r3); mpz_add(r2, r2, r4);
    }
    else if(test[0] == '+'){
	if(test[1] == 'A'){
            mpz_set_ui(r2, 1); mpz_mul_2exp(r2, r2, t * GMP_NUMB_BITS);
	    mpz_sub_ui(r3, r2, 1);
	    mpz_set_ui(q2, 1); mpz_mul_2exp(q2, q2, (u-t-2)*GMP_NUMB_BITS);
	    mpz_set_ui(q1, 1); mpz_mul_2exp(q1, q1, GMP_NUMB_BITS);
	    mpz_set_ui(A1, 1); mpz_mul_2exp(A1, A1, n * GMP_NUMB_BITS);
	    mpz_set_ui(B1, 1);
	}
	else if(test[1] == 'B'){
	    mpz_set_ui(r3, 1);
	    mpz_set_ui(r2, 1); mpz_mul_2exp(r2, r2, t * GMP_NUMB_BITS);
	    mpz_set_ui(q2, 1); mpz_mul_2exp(q2, q2, (u-t-1)*GMP_NUMB_BITS);
	    mpz_set_ui(q1, 1);
	    mpz_set_ui(A1, 1);
	    mpz_set_ui(B1, 0);
	}
    }
    mpz_mul(B0, q2, r2); mpz_add(B0, B0, r3);
    mpz_mul(A0, q1, B0); mpz_add(A0, A0, r2);
    mpz_mul_2exp(A, A0, n * GMP_NUMB_BITS);
    mpz_sub(A, A, A1);
    mpz_mul_2exp(B, B0, n * GMP_NUMB_BITS);
    mpz_add(B, B, B1);
    if(test_type == TEST_GCD){
	printf("Family(u=%d)\n", u);
	mpz_set_ui(r4, 1);
	TestGCD(r2, r3, A, B, NULL, r4, 0, test_type, test_algo, tot);
    }
    else if(test_type == TEST_HGCD){
	printf("HGCD(u=%d)\n", u);
	TestHGCD(A, B, gamma, test_algo, verbose, tot);
    }
    mpz_clears(A, B, A0, A1, B0, B1, q1, q2, q3, r2, r3, r4, NULL);
}

static void ThullYapTests(char *test, int test_type, int test_algo,
			  int umin, int umax, int verbose)
{
    double tot = 0.0;
    int u;

    for(u = umin; u < umax; u++){
	printf("##\n##### ThullYapFamily[%s]: u=%d\n##\n", test, u);
	ThullYapTest(test, test_type, test_algo, u, verbose, &tot);
    }
}

#define RECETTE_HGCD_NB 100

/* data[i] = {"label", "a", "b", "r", "params"}
   where r is the largest rem of (a, b) that is < sqrt(a).
 */
static void RecetteHGCD(int verbose, int algo, int imin, int imax)
{
    char *data[RECETTE_HGCD_NB][5] = {
	{"bad init size for prod[i] in vec_cut",
	 "31089351176298671056084108844115855960370929769037862721916295066616069417416702957707693799084715465642351794606855834607101701098741919439174619758708857185658736348932088009385294404206050447843942758360417473138462100026783583302567442099425782029789708477237727311402843797877855149735514626581030023941779607315148593302543256744976498649990924199724487087082551840499901116873016050886799061181618428874484947709381363832194940876863567413552556233011559894739410035656915600010816771018473767452635543886928651935828950598806632905995799737481234362771384263858948069987890403001705828276072963985175105923806918374800392282257386140225665117242679207434013222509069686388807666072684637759334529828295078844068449302044970620111442541949938277880655312610404732216435779193656967340586549561363249450357556932659374769623838227482305759074377368450601653356127438803217813985315956534811361856449444630169183558959123556913012968684697319811660485729367766326841703298497349675602214032866629569674084336280738101151753247150274820742839593629101728406667983536970315390713442188983116382822709543528484595053524044711089435725271627615035283007663867347469484437191781706551052718994480755092533041360669193571853364936372237201336090045164546444783499575022709186821299975825648647946719553379959060654817844871794085287961698555506528143524170858600896455004314982083158367713720160475660736375937516241915826775318051107204976962168324789610217243175479444182615861944602313434984224877730599815206129808421365675391577329249054188904096727867476984445350217651325425480894930952961627543833307779172584543806155312747125074457902209369815660514457289386831340412075789498428627587207368194380847336266080591806306462175842755798186681111429205098759431515839190170164087935874790947904620694892725163325818554223801574782393861358834835993825739537833690690291025262726383301663596990183342752265684194297248785675425739087037185449634374580476512512852467410753486404711919237829353869006612261019820688638738443694097580796499730071693196393564088731186356802870030442823790370063801214857687244805832458431479026967715489452790815519717943480559566902413731356874875669658345061447711914773259133508860355938663006902507041169627269196327901182144317711715494391360589469860596756580756986776631821651325225642289193314650794982319172852791357803407906108460362158669703504952455297581548683091105589715348606940995152160434844193555008437099259003997073198735546729328119661863171740709584473833041140862853793341328435302197872726426827094920581389857101810255610062555876848030042688952185959061046370826584672046205683107903035657745787672383653191511787941120551057091379309683996085476650049807886415606723005054596467149387530897555998292998921653085293017274489277465805549660705979376384649921321171352566798587503280327298586976175974602008110542069791152690423537767801602002914584646229625821795812932384583232484051518062992793995966987667567693180549849518066",
	 "3193916072198128228170420685463147136634914381544293783191127454683918987455336274623157179597065048780468604773179891703564618704513067297486912036939456852129277010422310610978831040590261160493620700875449063430637468564680681913520430484628021404692297862301057001240911604713599347903190413981422557209100535231633938604087139853382804982239678964606902683008016900520719963481617937090741198536300979029591531896609936804789230119515861956993317432710287354574155170902727895809718365681309624094235419480675484736086186311144785865717217134374369748718352616842805611126514620402658302020968566749236837413741947208909668870435728144005738271955750397296357559488179509291056039005066439858860132862773057367661355695948807750612087470273912159018398901043026619286556711367565557981353874364755461773665034721668557334604555030028180927713541518664335011656977218586982206654465941122381223294261253111572181342199881978142783830413249255610278315912049260317244362192148447999099005189752997294663365516613923891541497393652463625652759621126595543208727933346348576635462755209774833327055242076295077531525675289081501941983747270309622362791924602337633905445367784884882329328012107864537194774101371750823628970536850945365007114841946211878765130151067807607629637630195674516367141748713978262115097833794385263797885122588579324079542523183280823172253224320096909426703964527705322046989369784727044115053816754783061837283642599582233554238228506222291801971671307501841249719547153243304550867916905473984549727224288843714968958905540652028137914722283681182163094206181514005514461213952577071758148394710281183065227611790073252061276691065736926709388926968900119682075165974243776176726014325989125390741321240284060034919866546282185422571022900609749524699893599428955118136694836101677278943442769078196374727627539711798452077607900300459941036596291407998610125891934445482666518543583777369634899844266233036376482665682631086230345851789863127267204237559074955008917159998498439967051758659954611632712915456734610467387100388410943380872076136376858805697381681947834099690566286176845507964611294181307412761081984701441838646841430771756475097731503842328113812896671266914598781084701395287262196533566541316717513889438428690592697537750609498318588142489672800253508956406694212842649917532239676170531739501072672767329997422830958415737888336171926748684014248835156403653347353150334746571532509682262501107549823732344528684605597541371234489247920418178242353524341685963386179419677913157328694849519864859252074686765238390592500845547489043589876620276518455246188991287009078335841079385026712424859499117515953414099547649865223834243580639455751986856764501240700976393010009843784173386480612234010520813214358175108254437822208075011538154426367621190737196198729117798484122709034151123751589413192440890121332095980580567503724861292889828339498808614187928425096886819245217801742190973424310204893027054950366666088900744823897052271", "", "1/2"},
	{"small0+(-A): A=(BASE^6+BASE^7+BASE^8+BASE^9)*(BASE-1); B=BASE^9+BASE^8",
	 "4562440617622195218641171605700291324893228507248559930579192517899275167208637984499716416838159120738542165959789327571418239274707489704848415597950716977640580408105380034442067564256296960", "247330401473104534073910328949589787134705374099417686041542421913489618484255428614038165091316591966399545724183042767041396892493237260821300037913839737863040421154783232", "", "1/2"},
	{"(-B): A = sum(BASE^i, i=6..12)*(BASE-1); B:=BASE^12;",
	 "28638903918474961204418783933674838490721739172170652529441449702311064005352904159345284265824628375429359509218999720074396860757073337298438829647085367341472774164365407022527530796032310827951635762509973750265319139340482698699717980478483988480", "1552518092300708935148979488462502555256886017116696611139052038026050952686376886330878408828646477950487730697131073206171580044114814391444287275041181139204454976020849905550265285631598444825262999193716468750892846853816057856", "", "1/2"},
	{"(-B): three steps for leaf0101",
	 "528294531135665246380978688834991567724818628062097069553831427687307057641427757634462638699367785263651430503093027169149834037480910927793322845084352704201509202896001849580467321569020428701806177850432392523941586247934627645327842446241850793805873089050274955264","28638903918474961205971302025975547425870718660633155084698335719427760616492203527772808323045065764281257565237681329373985803294118512962709503310785925879127477984081636286460853908925844821134656658897650443511161098955555945534168240732439052288", "", "1/2"},
	{"(-B): carry for leaf01010",
	 "528294531135665246352339784916516606520399844128422231063109688515136405111986307932151574694014633773904673132734338291199454881071876076369824772487365264444971279629330738092784130859979973554549778917527397608357194874668575646641944346480011317022597695171651960832", "28638903918474961204418783933674838490721739172170652529441449702311064005353151489746757370358675470142449213811935277398499866750859960390717330873293387926239468945010381600279597058606828012156701341407865512635044345879027530213328960229616910336", "", "1/2"},
	{"(-B): leaf0100 (q1 > 1), clean",
	 "528294531135665246352339784916516606518847326036121522127960709026673902556724859474417255887657187894674394993257128678882347559502685537250538978462939576908386683999005084168731517676426441053024232908211188404148028292751561738838396898767036476489538580897737998335", "28638903918474961204418783933674838490721739172170652529441449702311064005352904159345284265824628375429359509218999720074396860757073376700445026041564579620512874307979212102266801261478978776245040008231745247475930553606737583615358787106474295296", "", "1/2"},
        {"(-B): leaf0101 (q3 = 1), clean",
	 "28638903918474961207523820118276256361019698149095657639955221736544457227631255565798859275731442650612135974066327807324473533991249586168881083354900740270571765259122817425665308884946694726684921944529248888336424098149272414285061087586255110144", "28638903918474961205971302025975547425870718660633155084698335719427760616492203527772808323045065764281257565237681329373985803294118512962709503310785925879127477984081636286460853908925844821134656658897650443511161098955555945534168240732439052288", "", "1/2"},
	 {"small0+(-C): A=(BASE^6+BASE^7+BASE^8+BASE^9)*(BASE-1); B=BASE^9",
	  "4562440617622195218641171605700291324893228507248559930579192517899275167208637984499716416838159120738542165959789327571418239274707489704848415597950716977640580408105380034442067564256296960", "247330401473104534060502521019647190035131349101211839914063056092897225106531867170316401061243044989597671426016139339351365034306751209967546155101893167916606772148699136", "", "1/2"},
	{"(-C): new version q3=2",
	 "1056589062271330492733318473751508174245218472190519300616941116202443462753404940685378969002945384024746176157711626098047881503284564176451452073231087197563740961734426653179420762446884382939747375155644888492213937133139879128752396595645431527526275208960576323584",
	 "57277807836949922410390085959650385916592457832803807614139785421738824621844860356716619484335633637189597427266645918099281452211277826607061632127243973632470035890999605343738057498978807458040345302095088939777124106407191635981610421066764648449",
	 "", "1/2"},
	{"(-D): q3=1, b1=1",
	 "528294531135665246380978688834991567724818628062097069553831427687307057641427757634462638699367785263651430503093027169149834037480910927793322845084352704201509202896001849580467321569020428701806177850432392523941586247934627645327842446241850793805873089050274955264",
	 "28638903918474961205971302025975547425870718660633155084698335719427760616492203527772808323045065764281257565237681329373985803294118512962709503310785925879127477984081636286460853908925844821134656658897650443511161098955555945534168240732439052289", "", "1/2"},
	{"(-D): q3=1 in new version (ok)",
	 "528294531135665246380978688834991567724818628062097069553831427687307057641427757634462638699367785263651430503093027169149834037480910927793322845084352704201509202896001849580467321569020428701806177850432392523941586247934627645327842446241850793805873089050274955264",
	 "28638903918474961205971302025975547425870718660633155084698335719427760616492203527772808323045065764281257565237681329373985803294118512962709503310785925879127477984081636286460853908925844821134656658897650443511161098955555945534168240732439052289", "", "1/2"},
	{"(+A): ap <= bp or more like leaf0101",
	 "9745314011399999081409971450146518803580906427428040739905601109285191735022347520518962870995459978174139428006722549312005013642649230399300643120937954231334153233202524057575377229037850752955921996029948774889754868398965219475363717470903475588084597508328146713725072538243092709376","9745314011399999080881676919010853557228566642511524133386753783249070212894386811492288968438735118699722172119065608747732091753926162222939315208625303825432715466579498543723083442263952200654360794754583576881720233162180047210555040543186220600456168155716212130403450564117503410176", "", "1/2"},
	{"(+B): leaf110 -- one step",
	 "528294531135665246352339784916516606520399844128422231063193850689561179088097601329660034557445045369545616727695464651854395740202985734329847167894384585968369399151441026482140394103440489507061931046264301499869887125263028811183872764204090162146825686130320474112","528294531135665246352339784916516606518847326036121522128044871201098676532840715312543337946305993331519565775009101173331447273971438830404357643043380992128018411500703935652139828592429381914834277944541618497747311871489616852409463518893185639965879272925510500352", "", "1/2"},
	{"(+B): leaf110 -- two steps?",
	 "2641472655678326231761698924582583032598894184457509737445250483598757020449303706611083925367330272155336121997846761997372172456819687955471192316754515283406303387573867476846536664611453051519775934757411147039721293743818108398343342305218195077429303191066139164672",
	 "7762590461503544675744897442312512776284430085583483055695260190130254763431616275495793192201240909252474536563105778714446052353019600728347001094604434756486237394140911459197525790994262503109239918312343412355535561288958607361", "", "1/2"},
	{"(+B): leaf110 -- new version + etiq6!!",
	 "528294531135665246352339784916516606520399844128422231063193850689561179088097601329660034557445045369545616727695478059662325682800085308354845373740512065334189991544818750043584115867470563054038732920562468403297577157121215297234726518086902108716772119779326558207", "528294531135665246352339784916516606518847326036121522128044871201098676532840715312543337946305993331519565775009101173331447273971438830404357643043380992128018411500703935652139828592429381914834277944541618497747311871489616852409463518893185639965879272925510500352", "", "1/2"},
	{"bad sizes in adjust_1_2; pb with b",
	 "8687694240707342124726711037162936851481100693958258939885624116731132652128078521023649234404148900080673004188924401800000290165507249533397226932773256148166595379346209425615024118695521283567405826297257732375416661002363125761825455866938223598655460569577543293856410483446599312903", "4919013702100521228528535487855653738571671335585261322916924953790373546099100301392244215492556356759645620502994946768410574246405388259265846764785735829423056570179205733347541913131950312048529306272423673046247900184494181440544717775018514839911628186765937119470902643969058241041", "", "1/2"},
	{"bad sizes in adjust_1_2; pb with a",
	 "7922717721938836114326349241959434345106190439150389939824882808720031050258268903878075358611538134728732595003655959692318762270011870059063969984412288246148137054512099457379454726324405019032680424688641894624290537690446013054931315794207926703307644512772971643897746679369705230419", "6535755275574621491872207996187667150283259255294854815584006383949106664084870072569549974616352814633332913398348896076546529927018371385388417305857596249253671573681556354375752621412330187907125779464615265463124577032366113994696805609916790802392692221478633738752517728186123313560", "", "1/2"},
	{"clear p0[rn+mn] and p1[rn+mn] before abs_sub_n",
	 "46612562666676684980040726197153735522725195572030812565607409892035363007451668482300424871224399460725100849065678356820492184894759970301319260412340996323811298609493639809820542027038266169884209050206492489472928834644419668180917740741837732772933195196131389891164780228361537024851168574860847152428170473737008650079362287479659801643077", "32507161110440110158657785885883568894611992648199462497677168601380722800546397113368506174397458193192247441523418376295420367341638313082321149684984882355521837612493893900437631124675782920636218944567523595010955627241204057717245388069933660280185516471279694015202074125799560558094903244945666151870309531911216957303352499075354805037985", "1/2"},
	{"pb of sizes in adjust_add_2_3",
	 "25532448831135911794981150222378549456272769425999485891375473051788205054360916498027975254486369994671820738782689669202601831300837222792383212375245128765371033712922220793749656616638737739454203440713886914", "11397854519656878756459789589871471695094104757043885017162425259160852685815610343374374299710513774819195056995202263312374605046677455099441880181587479578088042448820990460662262624123393326583795945363831058", "", "2/3"},
	{"two pbs in strassen: u1 of bad size, r1 of size rn+1",
	 "165813994435219829568956588029133412436420842814231861992207815250421972834250914399287543424687859037508623625590153826585543280523538775534365194896653365652887607893651963811424717321035206803544648964648944671654831812394583940926363173573391892149454713177917367259464463809546615094007522736555207590379", "142652427921443695429419179945069339692880853851418053704007139200629916892064469443677706792976454168467734240268912873099629715824470132878428175513232416246764299378657675165486675450163471366968528272212144164366932706918106404120443875130775130205773221213850147603658907699276526668121945312499845527534", "", "2/3"},
	{"w[1] == 0 instead of wn == 1",
	 "934145808888446410465244722877919328039295926296312353756908399224449172077979138992943688360806449434257951677074568349559281006266080938840158356200417962774337744526399059803715673124417578825398384676133584846718220152081130170195942442084199412736222622992385686050243928249763669880575606305465365515064922445939340336390265005351242866026466824166821017284383382401683061275452113050238912413963690031214935135155440741045192378713479536640835126127969124412329636315044528770724985523",
	 "139692152167065219266502359610617290326960319380299668332348834536821592851346368175709888627386935860795420579363925246360852797857174671499245361060756779091280520926979426455404769619240652912773340489535650115129416851965595575560410163904513133551405366841001124479620311511738850784974038590822624122297008015229409603370082048458972129509364653731982313053120755491352865150824166470939472462805618934474563214597753436589316583911362902069093641399259006059811954146610546164023876175", "", "2/3"},
	{"???",
	 "36264226582555288960475227397543462309500002663050251802533840436035840820220026588260470102699865387012448842175420346203348224853373679946649468778009461748449982271015707135695939409331011878428734827216541320227638508584838925264000258216236155924839046767617722335033321640654699714985605991053891181165060692936740106264073561379888623847820499114145058552750279254730254245719695516561856502336458448043936638995566028929217023077806402634318617434899266588963321478277077328890957628472567074052377001714179108083256828695927301731503337104305715584147987649966818089861", "5620294553458094101953134266414495056507804058310537932728218983302305243946346916627116952399382300086447252365397397801499882270227980821131419588475046163523982093011424727454217340895455837847108054675012285211607679297782462227794257372897800249967751924905935332675994221545888675203561879359800727951788945743728076713503595782284355801643370506680682942826101377748997360997838616128996892237707239586560263694455117087184462363790660315856725469274982049696003237247797309533219245127570056423960430614327993151710552841706514108849257845945493092226380807984495226653", "", "2/3"},
	{"(+A): ap <= bp -- revealed the small_1 problem",
	 "179769313486231590772930519078902473362854286956501987766134760727565709018538657784780720366663457439173461684984843076607624280589730503621393274416920260730490549249490786088105095134492145147157670525632802873441273084893193277489982381014397581692967864540020808215204104643703585471478600277875402735616",
	 "179769313486231590772930519078902473362325992425366322519782420942649192412019810458744598844535496730146787782428118217133207024702073563057120352528197192554129221337178135682203657367869119633305376738858904321139711883617828079481947745777612409428159187612303553227575675291091650888156978303749813436416",
	 "", "1/2"},
	{"ap < 0",
	 "4296408406945479606382838244475525030183939858178968543545195179632472498368071461228747772608041527005549116342113140053269042594110545829560981366249967228163816369062848061446211177098005967",
	 "2745660240965296315957985835259237594607720901382793298571020410265489358974220701814319746065836518173404497611287312417864057112097855697078855779722307825348468318187937579259879030741837940",
	 "777992250411945513254138364144051040963674649518961013606620960538097858172872642873075856290376",
	 "1/2"},
	{"bad use of second call to roll_back",
	 "347362628727297223593881359886854408717337769626076869682954867919950648982571501867068613876359136074109011883914448310508648878556906088409257942981350079059082695926721999818208638268320992492468209072461592400510981219261079481661601842100766515559586268197241018078358415325175682905295152960094508044456730718764963370053613038610538220359358467408325947580439220696895896411482927389207537226640314846794076900952625065753701368316499106958166255307934747229268823125262337",
	 "9870229388241628450235577501685946292375021850337596255817869284972229018590261122557808007583039754974834944635714454058676462044049941444624192634972829093406062594475582424327006781618085057494514791900120091545600439790789388272910053061650775770802441656268386484621343321903101146633818402896680554723331522664448853246610620224996393506642648284921985140587379820574657781007784731540648778065576286036129019336591859056651508176276068146306883764408457100288",
	 "", "1/2"},
	{"a:=BASE^9-BASE^6; b:=2*BASE^6; ||a||<9; m=6; a0:=BASE^3; ||a0||=3; t=3; ||a||<m+t, but ||a0||=t",
	 "247330401473104534060502521019647190035131349101211839914023654086700830627319588130216257447437965250327205979348191045947119312535253999356131888847008252275800144158392320",
	 "78804012392788958424558080200287227610159478540930893335896586808491443542994421222828532509769831281613255980613632",
	 "", "1/2"},
	{"26/03",
	 "8429109260574702676569247544215831222006128631858022916607904841203894722638595138635193801317530621221761060310005344384377503127366567901346189988261888354690005494388709501871219018227594411111633009702952763899734850436869123997501501554609791546592447586062496085114625751024412866578111187749732835137902958397136576333556972845715370513605119116280247081375912281860099433267135",
	 "7666398494737931348166473467430673618423262036078646447292435848103949738955159686898450679276985998470481106697878795326063268484038918172799518426871563632855131412284633737894766955628034436655218420177014488813763610457824166112624861997567317853025103523554058954444482940591773391492849964907056386507237685955124070465880483141328506380536385757232362603211895878770758415263727",
	 "2353336020041596871929203271142617125231187809658671919603201644135959628555883648722720080939841865355784040773330549974409594282962295033097791333632762568264706272789976129683681932980881147", "1/2"},
	{"XXX", "", "", "", ""}
    };
    mpz_t A, B, R;
    double tot = 0.0;
    char *gamma;
    int i;

    if(imax == -1)
	imax = imin+1;
    mpz_inits(A, B, R, NULL);
#if 0	    
    if(algo == ALGO_THULL_YAP){
	switch(family){
	case FAMILY_B1:
	    ThullYapTests("-B1", imin, imax, verbose);
	    return;
	case FAMILY_B2:
	    ThullYapTests("-B2", imin, imax, verbose);
	    return;
	ThullYapTests("-C", 6, 13, 0);
	ThullYapTests("-D", 6, 13, 0);
	ThullYapTests("+A", 6, 13, 0);
	ThullYapTests("+B", 6, 13, 0);
	}
    }
#endif
    for(i = imin; i < imax; i++){
	if(strcmp(data[i][0], "XXX") == 0)
	    break;
	if(verbose){
	    printf("////\n");
	    printf("//////// test %d: %s\n", i, data[i][0]);
	    printf("////\n");
	}
	mpz_set_str(A, data[i][1], 10);
	mpz_set_str(B, data[i][2], 10);
	mpz_set_str(R, data[i][2], 10);
	printf("// an=%lu, bn=%lu\n", mpz_size(A), mpz_size(B));
	gamma = data[i][4];
	TestHGCD(A, B, gamma, algo, verbose, &tot);
    }
    mpz_clears(A, B, R, NULL);
}

static void PrintFastGcdConfig(void)
{
    fprintf(stderr, "TY_LG_BASE                 =%d\n", TY_LG_BASE);
    fprintf(stderr, "QSET_FULL                  =%d\n", QSEQ_FULL);
    fprintf(stderr, "JEBELEAN_EUCLID_THRESHOLD  =%d\n", JEBELEAN_EUCLID_THRESHOLD);
#ifdef FINDTHRESHOLD
    fprintf(stderr, "ty_small_threshold         =%d\n", ty_small_threshold);
#else
    fprintf(stderr, "TY_SMALL_THRESHOLD         =%d\n", TY_SMALL_THRESHOLD);
#endif
    fprintf(stderr, "BUILD_PRODUCT_THRESHOLD1   =%d\n", BUILD_PRODUCT_THRESHOLD1);
    fprintf(stderr, "BUILD_PRODUCT_THRESHOLD2   =%d\n", BUILD_PRODUCT_THRESHOLD2);
}

static void PrintDefs(){
    /* print GMP thresholds from gmp-mparams.h */
#if 0 /* FIXME: name change? */
    fprintf(stderr, "MUL_KARATSUBA_THRESHOLD    =%3d\n", MUL_KARATSUBA_THRESHOLD);
    fprintf(stderr, "MUL_TOOM3_THRESHOLD        =%3d\n", MUL_TOOM3_THRESHOLD);
    fprintf(stderr, "MUL_TOOM4_THRESHOLD        =%3d\n", MUL_TOOM4_THRESHOLD);
    fprintf(stderr, "MUL_TOOM8H_THRESHOLD       =%3d\n", MUL_TOOM8H_THRESHOLD);
    fprintf(stderr, "MATRIX22_STRASSEN_THRESHOLD=%3d\n", MATRIX22_STRASSEN_THRESHOLD);
#endif
    /* params for Cornacchia */
    PrintFastGcdConfig();
}

static void usage_tests(char *s)
{
    fprintf(stderr, "Usage: %s \n", s);
    fprintf(stderr, "-verbose <level> verbose level # (default=1)\n");
    fprintf(stderr, "-algo [euclid|jebelean|thullyap|panwang|GMP]\n");
    fprintf(stderr, "-test <opt> where <opt> can be\n");
    fprintf(stderr, "  build           test build product\n");
    fprintf(stderr, "  qseq            test qseq data structure\n");
    fprintf(stderr, "  gcd             test fast gcd\n");
    fprintf(stderr, "  hgcd            test fast hgcd\n");
    fprintf(stderr, "  strassen        test Strassen\n");
    fprintf(stderr, "  matmul22        test for matrix multiplication\n");
    fprintf(stderr, "-recette [gcd|hgcd] recette for an algorithm\n");
    fprintf(stderr, "-family [mersenne|fibonnaci|random|random2| (default=random)\n");
    fprintf(stderr, "         mB1|mB2|mC|mD|pA|pB]\n");
    fprintf(stderr, "-imin <imin> -imax <imax>\n");
    fprintf(stderr, "\nExamples:\n");
    fprintf(stderr, "1. To use the recette of thullyap's hgcd:\n");
    fprintf(stderr, "\t%s -algo thullyap -recette hgcd\n", s);
    fprintf(stderr, "Same for only some of tests inside the recette:\n");
    fprintf(stderr, "\t%s -algo thullyap -recette hgcd -imin 3 -imax 5\n", s);
    fprintf(stderr, "2. To use one of the families\n");
    fprintf(stderr, "\t%s -test gcd -algo thullyap -imin 10 -imax 20"
	    " -family mersenne\n", s);
}

int main(int argc, char *argv[])
{
    mp_limb_t D = 1;
    int yone = 0;
    size_t sizemin, sizemax = 0, sizeincr = 1, sizemul = 0;
    double tt;
#if 0 /* obsolete? */
    char *gamma = NULL;
#endif
    int h, ntests, verbose = 0, test_type = 0, gmin = -1, gmax = -1;
#ifdef _OPENMP
    int numthreads = 1;
#endif
    int imin = -1, imax = -1, family = FAMILY_RANDOM, test_algo = ALGO_EUCLID;
    char *recette = NULL;

#ifndef hp
    setlinebuf(stdout);
#else
    {
	char stdout_buf[BUFSIZ], stderr_buf[BUFSIZ];
	
	setvbuf(stdout, stdout_buf, _IOLBF, BUFSIZ);
	setvbuf(stderr, stderr_buf, _IOLBF, BUFSIZ);
    }
#endif
    if(argc == 1){
	usage_tests(argv[0]);
	return 0;
    }
    fprintf(stderr, "#[%s] args:", GIT_VERSION);
    for(h = 0; h < argc; h++)
	fprintf(stderr, " %s", argv[h]);
    fprintf(stderr, "\n");
    ntests = 1;
    sizemin = 1;
    while(argc > 1 && argv[1][0] == '-'){
	if(argc > 2 && strcmp (argv[1], "-ntests") == 0){
	    ntests = atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-sizemin") == 0){
	    sizemin = (size_t)atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-sizemax") == 0){
	    sizemax = (size_t)atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-sizeincr") == 0){
	    sizeincr = (size_t)atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-sizemul") == 0){
	    sizemul = (size_t)atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-D") == 0){
	    D = (mp_limb_t)atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-yone") == 0){
	    yone = (mp_limb_t)atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-verbose") == 0){
	    verbose = atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
#if 0 /* obsolete? */	
	else if(argc > 2 && strcmp (argv[1], "-gamma") == 0){
	    gamma = argv[2];
	    argc -= 2;
	    argv += 2;
	}
#endif
	else if(argc > 2 && strcmp (argv[1], "-imin") == 0){
	    imin = atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-imax") == 0){
	    imax = atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-gmin") == 0){
	    gmin = atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-gmax") == 0){
	    gmax = atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
#ifdef _OPENMP
	else if(argc > 2 && strcmp (argv[1], "-numthreads") == 0){
	    numthreads = atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
#endif
	else if(argc > 2 && strcmp (argv[1], "-recette") == 0){
	    recette = argv[2];
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-family") == 0){
	    if(strcmp(argv[2], "mersenne") == 0)
		family = FAMILY_MERSENNE;
	    else if(strcmp(argv[2], "fibonacci") == 0)
		family = FAMILY_FIBONACCI;
	    else if(strcmp(argv[2], "random") == 0)
		family = FAMILY_RANDOM;
	    else if(strcmp(argv[2], "random2") == 0)
		family = FAMILY_RANDOM2;
	    else if(strcmp(argv[2], "mB1") == 0)
		family = FAMILY_mB1;
	    else if(strcmp(argv[2], "mB2") == 0)
		family = FAMILY_mB2;
	    else if(strcmp(argv[2], "mC") == 0)
		family = FAMILY_mC;
	    else if(strcmp(argv[2], "mD") == 0)
		family = FAMILY_mD;
	    else if(strcmp(argv[2], "pA") == 0)
		family = FAMILY_pA;
	    else if(strcmp(argv[2], "pB") == 0)
		family = FAMILY_pB;
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-test") == 0){
	    if(strcmp(argv[2], "build") == 0)
		test_type = TEST_BUILD_PRODUCT;
	    else if(strcmp(argv[2], "qseq") == 0)
		test_type = TEST_QSEQ;
	    else if(strcmp(argv[2], "gcd") == 0)
		test_type = TEST_GCD;
	    else if(strcmp(argv[2], "hgcd") == 0)
		test_type = TEST_HGCD;
	    else if(strcmp(argv[2], "strassen") == 0)
		test_type = TEST_STRASSEN;
	    else if(strcmp(argv[2], "matmul22") == 0)
		test_type = TEST_MATMUL22;
	    else if(strcmp(argv[2], "matmulvec") == 0)
		test_type = TEST_MATMULVEC;
	    else if(strcmp(argv[2], "cornacchia") == 0)
		test_type = TEST_CORNACCHIA;
	    else{
		fprintf(stderr, "Unknown args: %s\n", argv[2]);
		usage_tests(argv[0]);
		exit(1);
	    }
	    argc -= 2;
	    argv += 2;
	}
	else if(argc > 2 && strcmp (argv[1], "-algo") == 0){
	    if(strcmp(argv[2], "euclid") == 0)
		test_algo = ALGO_EUCLID;
	    else if(strcmp(argv[2], "qeuclid") == 0)
		test_algo = ALGO_QEUCLID;
	    else if(strcmp(argv[2], "jebelean") == 0)
		test_algo = ALGO_JEBELEAN;
	    else if(strcmp(argv[2], "qjebelean") == 0)
		test_algo = ALGO_QJEBELEAN;
	    else if(strcmp(argv[2], "thullyap") == 0)
		test_algo = ALGO_THULL_YAP;
	    else if(strcmp(argv[2], "panwang") == 0)
		test_algo = ALGO_PAN_WANG;
	    else if(strcmp(argv[2], "gmp") == 0)
		test_algo = ALGO_GMP;
	    else{
		fprintf(stderr, "Unknown arg: %s\n", argv[2]);
		usage_tests(argv[0]);
		exit(1);
	    }
	    argc -= 2;
	    argv += 2;
	}
#ifdef FINDTHRESHOLD
	else if(argc > 2 && strcmp(argv[1], "-threshold") == 0){
	    ty_small_threshold = atoi(argv[2]);
	    argc -= 2;
	    argv += 2;
	}
#endif
	else{
	    fprintf(stderr, "Error, unknown option %s\n", argv[1]);
	    usage_tests(argv[0]);
	    exit(1);
	}
    }
    if(imin == -1 && imax == -1){
	imin = 0;
	imax = 100;
    }
    if(imax == -1) imax = imin+1;
    if(gmin == -1) gmin = 2;
    if(gmax == -1) gmax = gmin+1;
    if(sizemax == 0)
	sizemax = sizemin + sizeincr;
    assert(sizemul == 0 || sizemul > 1);
    PrintDefs();
    if(recette != NULL){
	if(strcmp(recette, "gcd") == 0)
	    RecetteGcd(verbose, test_algo, imin, imax);
	else if(strcmp(recette, "hgcd") == 0)
	    RecetteHGCD(verbose, test_algo, imin, imax);
    }
    else if((test_type == TEST_GCD || test_type == TEST_HGCD)
	    && family != FAMILY_NONE){
	tt = runtime();
	switch(family){
	case FAMILY_FIBONACCI:
	    UseFibonacci(test_type, test_algo, imin, imax, verbose);
	    break;
	case FAMILY_MERSENNE:
	    UseMersenne(test_type, test_algo, imin, imax, verbose);
	    break;
	case FAMILY_RANDOM:
	case FAMILY_RANDOM2:
	    UseRandom(test_type, family, test_algo, sizemin, sizemax, sizeincr,
		      sizemul, ntests, verbose);
	    break;
	case FAMILY_mB1:
	    ThullYapTests("-B1", test_type, test_algo, imin, imax, verbose);
	    break;
	case FAMILY_mB2:
	    ThullYapTests("-B2", test_type, test_algo, imin, imax, verbose);
	    break;
	case FAMILY_mC:
	    ThullYapTests("-C", test_type, test_algo, imin, imax, verbose);
	    break;
	case FAMILY_mD:
	    ThullYapTests("-D", test_type, test_algo, imin, imax, verbose);
	    break;
	case FAMILY_pA:
	    ThullYapTests("+A", test_type, test_algo, imin, imax, verbose);
	    break;
	case FAMILY_pB:
	    ThullYapTests("+B", test_type, test_algo, imin, imax, verbose);
	    break;
	default:
	    fprintf(stderr, "Unkown family: %d\n", family);
	}
	fprintf(stderr, "Time for %d: %lf s\n", family, (runtime()-tt)/1000.);
    }
    else{
	if(test_type == TEST_MATMUL22)
	    TestMatMul22(sizemin, sizemax, sizeincr, sizemul,
			 gmin, gmax, ntests);
	else if(test_type == TEST_MATMULVEC)
	    TestMatMulVec(sizemin, sizemax, sizeincr, sizemul,
			  gmin, gmax, ntests);
#ifdef REGULAR
	else if(test_type == TEST_BUILD_PRODUCT)
	    TestBuildProduct(sizemin, sizemax, sizeincr, sizemul, ntests);
#endif
	else if(test_type == TEST_QSEQ)
	    TestQseq();
	else if(test_type == TEST_CORNACCHIA)
	    TestCornacchia(test_algo, family, D, yone,
			   sizemin, sizemax, sizeincr,
			   sizemul, ntests, verbose);
#if ONE_DAY
	else if(test_type == TEST_STRASSEN)
	    TestStrassen(sizemin, sizemax, sizeincr, sizemul, ntests);
#endif
    }
    return 0;
}
