/*! \file
    \brief Handling quotient matrices [[q, 1], [1, 0]]
*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>

#include "gmp.h"
#include "gmp-impl.h"
#include "longlong.h"

#include "gcd_matrix.h"
#include "qseq.h"
#include "regular.h"
#include "gcd_utils.h"

#define DEBUG_QSEQ 0
#define STATS_QSEQ 0

/* For a of size n */
void qseq_init(qseq_t S, mp_size_t n)
{
    size_t sz = ((size_t) n) * GMP_NUMB_BITS;
    
#if DEBUG_QSEQ >= 1
    printf("qseq_init(%lu)\n", sz);
#endif
    if(n != 0){
	S->alloc_large = 8;
	S->tab_large = (mp_limb_t *)malloc(S->alloc_large * sizeof(mp_limb_t));
	S->tab_large_nl = (mp_size_t *)malloc(S->alloc_large*sizeof(mp_size_t));
	S->tind = (int *)malloc(sz * sizeof(int));
    }
    else{
	S->alloc_large = 0;
	S->tab_large = NULL;
	S->tab_large_nl = NULL;
	S->tind = NULL;
    }
    S->alloc_ind = sz;
    S->first_large = 1;
    S->last_large = 1;
    S->first = 0;
    S->last = 0;
}

void qseq_realloc(MAYBE_UNUSED qseq_t S, MAYBE_UNUSED size_t sz){
    assert(0);
}

void qseq_realloc_ind(qseq_t S, MAYBE_UNUSED size_t sz){
    do {
	S->alloc_ind <<= 1;
    } while(sz > S->alloc_ind);
#if DEBUG_QSEQ >= 1
    fprintf(stderr, "qseq_realloc_ind: %lu -> %lu\n", sz, S->alloc_ind);
#endif
    S->tind = realloc(S->tind, S->alloc_ind * sizeof(mp_size_t));
}

void qseq_realloc_large(qseq_t S, MAYBE_UNUSED size_t sz){
    do {
	S->alloc_large <<= 1;
    } while(sz > S->alloc_large);
#if DEBUG_QSEQ >= 1
    fprintf(stderr, "qseq_realloc_large: %lu -> %lu\n", sz, S->alloc_large);
#endif
    S->tab_large = realloc(S->tab_large, S->alloc_large * sizeof(mp_limb_t));
    S->tab_large_nl = realloc(S->tab_large_nl,
			      S->alloc_large * sizeof(mp_size_t));
}

#if 0
static void qseq_check_alloc(qseq_t S)
{
    size_t i;

    printf("qseq_check_alloc: %p\n", S);
    for(i = 0; i < S->last; i++)
	assert(S->tabq[i]->_mp_alloc >= 0);
}
#endif

void qseq_print_cell(qseq_t S, int i)
{
#if QSEQ_DATA_TYPE == QSEQ_DATA_TAB1 || QSEQ_DATA_TYPE == QSEQ_DATA_TAB2
# if TRICK_FOR_ONE == 1
    if(qseq_is_cell_one(S, i) != 0)
	printf("1");
    else
# endif	    
	MPN_PRINT(qseq_n(S, i), qseq_nl(S, i));
#elif QSEQ_DATA_TYPE == QSEQ_DATA_TAB3
    int j = S->tind[i];

    if(j >= 0)
	printf("%d", j);
    else{
	j = -j;
	MPN_PRINT(S->tab_large+j, S->tab_large_nl[j]);
    }
#endif
}

MAYBE_UNUSED
void qseq_print(qseq_t S)
{
    int i;

    printf("[%d..%d[<", S->first, S->last);
    for(i = S->first; i < S->last; i++){
	if(i > S->first)
	    printf(", ");
	qseq_print_cell(S, i);
    }
    printf(">");
}

MAYBE_UNUSED
void qseq_dump(qseq_t S)
{
    int i;
    
    printf("alloc_large = %lu; first_large=%d; last_large=%d\n",
	   S->alloc_large, S->first_large, S->last_large);
    for(i = S->first_large; i < S->last_large; i += S->tab_large_nl[i]){
	printf("large[%d]=[%lu]", i, S->tab_large_nl[i]);
	MPN_PRINT(S->tab_large+i, S->tab_large_nl[i]);
	printf("\n");
    }
    printf("alloc_ind = %lu; first=%d; last=%d\n",
	   S->alloc_ind, S->first, S->last);
    for(i = S->first; i < S->last; i++){
	printf("C[%d]=", i); qseq_print_cell(S, i); printf("\n");
    }
}

/* store large */
static void qseq_set_n_large(qseq_t S, int i, mp_ptr a, mp_size_t an)
{
    mp_size_t tmpn;

#if STATS_QSEQ >= 1
    printf("Store large: [%lu] ", an); MPN_PRINT(a, an); printf("\n");
#endif
    S->tind[i] = -S->last_large;
    tmpn = S->last_large + an;
    if(tmpn > (int)S->alloc_large)
	qseq_realloc_large(S, tmpn);
    MPN_COPY(S->tab_large+S->last_large, a, an);
    S->tab_large_nl[S->last_large] = an;
    S->last_large += an;
}

static void qseq_set_n(qseq_t S, int i, mp_ptr a, mp_size_t an)
{
    mp_limb_t l = ((mp_limb_t)1) << 31;

    if(i >= (int)S->alloc_ind)
	qseq_realloc_ind(S, i);
    if(an == 1 && a[0] < l)
	S->tind[i] = (int)a[0];
    else
	qseq_set_n_large(S, i, a, an);
}

void qseq_copy_n(mp_ptr n, qseq_t S, int i)
{
    int j = S->tind[i];

    if(j >= 0)
	n[0] = (mp_limb_t)j;
    else{
	j = -j;
	MPN_COPY(n, S->tab_large+j, S->tab_large_nl[j]);
    }
}

void qseq_add_last_mpn(qseq_t S, mp_ptr a, mp_size_t an)
{
#if TRICK_FOR_ONE == 1
    int isone = (an == 1) && (a[0] == 1);
#endif
#if DEBUG_QSEQ >= 1
    printf("qseq_add_last_mpn: adding "); MPN_PRINT(a, an); printf("\n");
#endif
    assert(an > 0);
    if(an == 1 && a[0] == 0)
	return;
    qseq_set_n(S, S->last, a, an);
    S->last += 1;
#if DEBUG_QSEQ >= 1
    if(an > 1){
	printf("Adding large[%lu] q:=", an); MPN_PRINT(a, an); printf(";\n");
	MPN_NORMALIZE(a, an);
	assert(an > 1);
    }
#endif
}

/* FIXME: do better -> q[0] = 0 */
void qseq_add_last_one(qseq_t S)
{
    mp_limb_t q = 1;

    qseq_add_last_mpn(S, &q, 1);
}

static int qseq_are_cells_equal(qseq_t S, int i, qseq_t R, int j)
{
#if 0 /* blourk */
    int Sone = qseq_is_cell_one(S, i), Rone = qseq_is_cell_one(R, j);

    if(Sone != Rone)                   return 0;
    if(Sone != 0)                      return 1;
    if(qseq_nl(S, i) != qseq_nl(R, j)) return 0;
    return mpn_cmp(qseq_n(S, i), qseq_n(R, j), qseq_nl(R, j)) == 0;
#else
    int Sk = S->tind[i], Rk = R->tind[j];
    assert(Sk != 0 && Rk != 0);
    if(Sk > 0)
	/* Sk and Rk should be small and equal */
	return (Sk == Rk);
    /* Sk < 0 */
    if(Rk > 0)
	return 0;
    /* both are large */
    if(qseq_nl(S, i) != qseq_nl(R, j))
	return 0;
    return mpn_cmp(qseq_n(S, i), qseq_n(R, j), qseq_nl(R, j)) == 0;
#endif
}

/* FIXME: first or not? */
int qseq_is_equal(qseq_t S, qseq_t R)
{
    if(qseq_card(S) != qseq_card(R)){
	printf("#S=%d #R=%d\n", qseq_card(S), qseq_card(R));
	return 0;
    }
#if QSEQ_DATA_TYPE == QSEQ_DATA_TAB1 || QSEQ_DATA_TYPE == QSEQ_DATA_TAB2 || QSEQ_DATA_TYPE == QSEQ_DATA_TAB3
    int i, j;

#if DEBUG_QSEQ > 0
    printf("Sfirst=%d, Rfirst=%d\n", S->first, R->first);
#endif
    for(i = S->first, j = R->first; i < S->last; i++, j++)
	if(qseq_are_cells_equal(S, i, R, j) == 0){
	    printf("S[%d] != R[%d]\n", i, j);
	    printf("S[%d]:=", i); qseq_print_cell(S, i); printf(";\n");
	    printf("R[%d]:=", j); qseq_print_cell(R, j); printf(";\n");
	    return 0;
	}
#endif
    return 1;
}

void qseq_remove_last(qseq_t S)
{
    int j;
    
    assert(S->last > S->first);
#if DEBUG_QSEQ >= 1
    printf("qseq_remove_last: removing "); qseq_print_cell(S, S->last-1);
    printf("\n");
#endif
    j = S->tind[S->last-1];
    if(j < 0)
	S->last_large -= S->tab_large_nl[-j];
    S->last -= 1;
}

void qseq_clear(MAYBE_UNUSED qseq_t S)
{
#if QSEQ_DATA_TYPE == QSEQ_DATA_TAB1 || (QSEQ_DATA_TYPE == QSEQ_DATA_TAB3 && STATS_QSEQ >= 1)
    int i;
#endif
    if(S->alloc_ind == 0)
	return;
# if STATS_QSEQ >= 1
    fprintf(stderr, "small: %d large: %d\n",
	    qseq_card(S)-qseq_card_large(S), qseq_card_large(S));
    for(i = S->first; i < S->last; i++)
	fprintf(stderr, "S %d\n", S->tind[i]);
# endif
    free(S->tab_large);
    free(S->tab_large_nl);
    free(S->tind);
    S->tab_large = NULL;
    S->tind = NULL;
    S->alloc_large = 0;
    S->alloc_ind = 0;
    S->last = 0;
}

MAYBE_UNUSED
static int qseq_length(qseq_t S)
{
    return S->last;
}

/** <q> = [[q, 1], [1, 0]] -> inverse is [[0, 1], [1, -q]]. 
    [[0, 1], [1, -q]] [a, b] = [b, a-b*q]
    SIDE-EFFECT: (ap, bp) = M^{-1} (a, b); i.e., we replay Euclid's descent.
    REM: (ap, bp) will be much smaller than (a, b).
*/
MAYBE_UNUSED
static void qseq_apply_inverse_plain(mpz_t ap, mpz_t bp, qseq_t S, 
				     mpz_t a, mpz_t b)
{
#if TIMINGS >= 2
    double tt = runtime();
#endif
    mpz_t tp;
    int i, j;

    mpz_init_set_ui(tp, 0);
    mpz_set(ap, a);
    mpz_set(bp, b);
    for(i = S->first; i < S->last; i++){
#if DEBUG_QSEQ >= 1
	gmp_printf("ap:=%Zd;\nbp:=%Zd;\n", ap, bp);
#endif
	j = S->tind[i];
	if(j > 0){
	    mp_limb_t q = ((mp_limb_t)j);
	    MPZ_SET_MPN(tp, &q, 1);
	}
	else if(j < 0){
	    j = -j;
	    MPZ_SET_MPN(tp, S->tab_large+j, S->tab_large_nl[j]);
	}
#if DEBUG_QSEQ >= 1
	gmp_printf("q:=%Zd;\n", tp);
#endif
	mpz_submul(ap, bp, tp);
#if DEBUG_QSEQ >= 1
	gmp_printf("ap-q*bp eq %Zd;\n", ap);
#endif
	mpz_set(tp, ap);
	mpz_set(ap, bp);
	mpz_set(bp, tp);
	assert(mpz_sgn(ap) > 0);
	assert(mpz_sgn(bp) >= 0); /* bp can be 0 */
    }
#if TIMINGS >= 2
    printf("#T# qseq_apply_inverse: %d %lf\n", S->last, runtime()-tt);
#endif
    mpz_clear(tp);
}

/** Do we have (ap; bp) == Q^(-1) (a; b)? */
MAYBE_UNUSED
void qseq_check(qseq_t S,
		mp_ptr a, mp_size_t an,
		mp_ptr b, mp_size_t bn,
		mp_ptr ap, mp_size_t apn,
		mp_ptr bp, mp_size_t bpn)
{
    mpz_t za, zb, zap, zbp;
    int status = 1;
    
    mpz_inits(zap, zbp, NULL);
    mpz_init_set_ui(za, 0);
    MPZ_SET_MPN(za, a, an);
    mpz_init_set_ui(zb, 0);
    MPZ_SET_MPN(zb, b, bn);
    qseq_apply_inverse_plain(zap, zbp, S, za, zb);
    MPN_NORMALIZE(ap, apn);
    MPN_NORMALIZE(bp, bpn);
    if(apn != SIZ(zap) || mpn_cmp(ap, PTR(zap), apn) != 0){
	printf("Error in qseq_check:\n");
	gmp_printf("zap:=%Zd;\n", zap);
	printf("ap:="); MPN_PRINT(ap, apn); printf(";\n");
	status = 0;
    }
    if(bpn != SIZ(zbp) || mpn_cmp(bp, PTR(zbp), bpn) != 0){
	printf("Error in qseq_check:\n");
	gmp_printf("zbp:=%Zd;\n", zbp);
	printf("bp:="); MPN_PRINT(bp, bpn); printf(";\n");
	status = 0;
    }
    assert(status != 0);
    mpz_clears(za, zb, zap, zbp, NULL);
}

int qseq_determinant(qseq_t S)
{
    int det = (int)((S->last - S->first) & 1);
    return (det == 0 ? 1 : -1);
}

/** INPUT: *xn is the current of digits of x; size(x) must be large enough(!).
    (x, *xn) += (y, yn) * last(S).
*/
void mpn_qseq_addmul_last(mp_ptr x, mp_size_t *xn,
			  mp_ptr y, mp_size_t yn, qseq_t S)
{
    mp_limb_t q = 1;

    int j = S->tind[S->last-1];
    if(j == 0)
	return;
    else if(j > 0){
	q = (mp_limb_t)j;
	gcd_addmul(x, xn, y, yn, &q, 1);
    }
    else{
	j = -j;
	gcd_addmul(x, xn, y, yn, S->tab_large+j, S->tab_large_nl[j]);
    }
}

mp_ptr mpn_qseq_get_last(mp_size_t *tmpn, qseq_t S)
{
    mp_ptr tmp;
    int j = S->tind[S->last-1];
    if(j >= 0){
	tmp = (mp_ptr)malloc(sizeof(mp_limb_t));
	tmp[0] = (mp_limb_t)j;
	*tmpn = 1;
    }
    else{
	j = -j;
	tmp = (mp_ptr)malloc(S->tab_large_nl[j] * sizeof(mp_limb_t));
	MPN_COPY(tmp, S->tab_large+j, S->tab_large_nl[j]);
	*tmpn = S->tab_large_nl[j];
    }
    return tmp;
}

/* Q *= [[q_i, 1], [1, 0]] */
void qseq_mul_qi(struct hgcd_matrix *Q, qseq_t lq, int i,
		 mp_ptr tp, mp_size_t tp_alloc)
{
    mp_ptr q = NULL;
    mp_size_t qn;
    mp_limb_t tmp = 0;

    int j = lq->tind[i];
    if(j >= 0){
	tmp = (mp_limb_t)j;
	q = &tmp;
	qn = 1;
    }
    else{
	q = qseq_n(lq, i);
	qn = qseq_nl(lq, i);
    }
    hgcd_matrix_mul_q(Q, q, qn, tp, tp_alloc);
}

/* Q <- [[q_i, 1], [1, 0]] */
void qseq_init_matrix_set_q(struct hgcd_matrix *Q, qseq_t lq, int i)
{
    mp_ptr q = NULL;
    mp_size_t qn;
    mp_limb_t tmp = 0;

    int j = lq->tind[i];
    qn = (j >= 0 ? 1 : qseq_nl(lq, i));
    if(j >= 0)
	tmp = j;
    else{
	q = qseq_n(lq, i);
	qn = qseq_nl(lq, i);
    }
    hgcd_matrix_init(Q, qn);
    if(tmp > 0)
	Q->p[0][0][0] = tmp;
    else
	MPN_COPY(Q->p[0][0], q, qn);
    Q->n = qn;
    Q->p[0][1][0] = 1;
    Q->p[1][0][0] = 1;
}

/* S <- S omul T by the most stupid algo */
static void qseq_omul_naive(qseq_t S, qseq_t T)
{
    mp_limb_t q;
    int i;
    
    for(i = T->first; i < T->last; i++){
	int j = T->tind[i];
	if(j == 0)
	    return; /* should not happen! */
	else if(j > 0){
	    q = (mp_limb_t)j;
	    qseq_add_last_mpn(S, &q, 1);
	}
	else{
	    j = -j;
	    qseq_add_last_mpn(S, T->tab_large+j, T->tab_large_nl[j]);
	}
    }   
}

/* HERE: realloc + memcpy? + special treatment large q's from T?
   but this would require localizing negative indices in T, for a |T| cost?
 */
static void qseq_omul_cpy(qseq_t S, qseq_t T)
{
    int i, cardT = qseq_card(T);
    size_t Slast = S->last, sz = S->last + cardT;

#if 0
    /* enlarge S->tind */
    S->alloc_ind += cardT;
    S->tind = realloc(S->tind, S->alloc_ind * sizeof(mp_size_t));
#else
    if(S->alloc_ind < sz)
	/* enlarge only if needed */
	qseq_realloc_ind(S, sz);
#endif
    /* recopy T->tind */
    memcpy(S->tind + S->last, T->tind + T->first, cardT * sizeof(int));
    S->last += cardT;
    /* explore for negative indices */
    for(i = Slast; i < S->last; i++)
	if(S->tind[i] < 0){
	    /* assign large quotient to S */
	    int j = -S->tind[i];
#if DEBUG_QSEQ >= 1
	    printf("Adding a large quotient from T\n");
#endif
	    qseq_set_n_large(S, i, T->tab_large+j, T->tab_large_nl[j]);
	    /* S->tind[i] will have been updated to a new < 0 value */
	}
}

/* S <- S omul T. */
void qseq_omul(qseq_t S, qseq_t T)
{
    int cardT;
    
    /* we should take care to S = [] */
    if(qseq_card(S) == 0)
	printf("Case S = []; card(T) = %d\n", qseq_card(T));
    cardT = qseq_card(T);
#if DEBUG_QSEQ >= 1
    printf("#S = %d, #T = %d\n", qseq_card(S), cardT);
#endif
    if(cardT <= 10)
	qseq_omul_naive(S, T);
    else if(cardT <= 2000000)
	qseq_omul_cpy(S, T);
    else
	assert(0);
}
