/*! \file
    \brief Common functions for gcd algorithms
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <assert.h>

#include "gmp.h"
#include "gmp-impl.h"

#include "gcd_utils.h"
#include "qseq.h"
#include "gcd_matrix.h"
#include "regular.h"
#include "gcd_common.h"
#include "gcd_euclid.h"
#include "gcd_jebelean.h"

#define TIMINGS      0
#define DEBUG_COMMON 0

/* 
   R->Q = prod(Q[i], i in [Rfirst..R->last[
   We have two strategies depending on nq = R->lq->last - Rfirst.
   Rem: we can arrive from gcd_choose where nothing was performed yet...
   Rem: better to handle memory inside this function.

   The two cases R = Id and nq = 0 are treated to speed up computations.

*/
void gcd_mul_M(regular_t R, int Rfirst)
{
    mp_ptr tp;
    mp_size_t tp_alloc, tpn;
#if TIMINGS >= 1
    double tt, tot = runtime();
#endif
    int nq = R->lq->last - Rfirst, i;
    int svg_first = R->lq->first;

#if DEBUG_COMMON >= 1
    printf("gcd_mul_M: R->Q *= lq[%d..%d[\n", Rfirst, R->lq->last);
#endif
#if DEBUG_COMMON >= 1
    int svg_last = R->lq->last;
    printf("----- check at the beginning of mul_M\n");
    R->lq->last = Rfirst;
    check_Q_from_lq(R->Q, R->lq);
    R->lq->last = svg_last;
#endif
    if(nq == 0)
	/* M *= Id */
	return;
#if TIMINGS >= 1
    tt = runtime();
#endif
#if DEBUG_COMMON >= 1
    printf("Morig:="); hgcd_matrix_print(R->Q);
#endif
    if(nq <= 10){
	tp_alloc = R->Q->n << 1;
	tp = (mp_ptr)malloc(tp_alloc * sizeof(mp_limb_t));
	for(i = Rfirst; i < R->lq->last; i++){
	    tpn = R->Q->n + qseq_nl(R->lq, i) + 1;
	    if(tpn > tp_alloc){
		do {
		    tp_alloc <<= 1;
		} while(tp_alloc < tpn);
		tp = realloc(tp, tp_alloc * sizeof(mp_limb_t));
	    }
	    regular_mul_qi(R, i, tp, tp_alloc);
	}
	free(tp);
    }
    else{
	struct hgcd_matrix MS;

	R->lq->first = Rfirst;
#if TIMINGS >= 1
	tt = runtime();
#endif
	qseq_build_product(&MS, R->lq);
#if TIMINGS >= 1
	fprintf(stderr, "qseq_build_product: %d %lu %lf\n",
		qseq_card(R->lq), MS.n, runtime()-tt);
#endif
	assert(MS.n <= MS.alloc);
#if DEBUG_COMMON >= 1
	printf("MS:="); hgcd_matrix_print(&MS);
#endif
	R->lq->first = svg_first;
	/* M *= MS */
	if(hgcd_matrix_is_identity(R->Q) != 0)
	    hgcd_matrix_set(R->Q, &MS);
	else{
#if TIMINGS >= 1
	    tt = runtime();
#endif
	    hgcd_matrix_mul(R->Q, &MS);
#if TIMINGS >= 1
	    fprintf(stderr, "matrix_mul: %lu %lu %lf\n",
		    R->Q->n, MS.n, runtime()-tt);
#endif
	}
	hgcd_matrix_clear(&MS);
    }
#if TIMINGS >= 1
    fprintf(stderr, "mul_M: %d %lf\n", nq, runtime()-tot);
#endif
#if DEBUG_COMMON >= 1
    printf("----- check at the end of mul_M\n");
    R->lq->first = svg_first;
    check_Q_from_lq(R->Q, R->lq);
#endif
}

/** INPUT: an >= bn; b[bn..an[ cleared 
           tp has size >= 2*an+3
    PRECONDITION: R->Q == prod Q[Q->first..Q->last[
    SIDE-EFFECT: 
	a[an] and b[an] contain the final lengths.
	R is updated.
    POSTCONDITION: newQ == newlq.
        if qgcd == 0:
            a >= rmin > b      if m = 0
            ||a|| >= m > ||b|| if m > 0.
	elif qgcd == 1:
	    ||Q_i|| <= base^m < ||Q_{i+1}||
	a[an] and b[an] contain the new sizes.
        R->Q == prod Q[Q->first..Q->last[
*/
void gcd_small(regular_t R, mp_ptr a, mp_size_t an, mp_ptr b, mp_size_t bn,
	       mp_size_t m, mp_ptr rmin, mp_size_t rminn,
	       mp_ptr tp, mp_size_t tp_alloc, int qgcd)
{
#if TIMINGS >= 1
    double tt = runtime();
#endif
    if(qseq_is_used(R->lq) != 0){
#if DEBUG_COMMON >= 100
	printf("----- check on entering gcd_small\n");
	check_Q_from_lq(R->Q, R->lq);
#endif
	ASSERT(R->det == qseq_determinant(R->lq));
    }
    if(1) /* TODO: fix the pb of losing q's; also qgcd == 0) */
	/* TODO: make this work for qgcd == 1 */
	jebelean_gcd(R, a, an, b, bn, m, rmin, rminn, tp, tp_alloc, qgcd);
    else
	euclid_gcd(R, a, an, b, bn, m, rmin, rminn, tp, tp_alloc, qgcd);
#if TIMINGS >= 1
    fprintf(stderr, "small_gcd: %lu %lu %lf\n", an, bn, runtime()-tt);
#endif
    if(qseq_is_used(R->lq) != 0)
	ASSERT(R->det == qseq_determinant(R->lq));
#if 0 /* FIXME: PanWang could suffer from this */
    if(qgcd == 0){
	/* update R->Q; already done if qgcd == 1 */
	if(qseq_is_used(R->lq) != 0){
	    gcd_mul_M(R, Rfirst);
	}
	else
	    assert(0); /* what is to be done here, actually? */
    }
#endif
}
